/** rti2code.cpp  Utility to generate registry initialisation code from rti files */

/* FAU Discrete Event Systems Library (libfaudes)

Copyright (C) 2009 Ruediger Berndt
Copyright (C) 2010, 2023, 2024 Thomas Moor

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA */
	 

#include <string>
#include <iostream>
#include <fstream>
#include "corefaudes.h"


using namespace faudes;

// ******************************************************************
// error exit
// ******************************************************************

void Usage(const std::string& rMessage="") {
  // UI hints
  if(rMessage!="") {
    std::cerr << rMessage << std::endl;
    std::cout << "" << std::endl;
  }
  std::cerr << "rti2code: " << VersionString()  << std::endl;
  std::cerr << std::endl;
  std::cerr << "utility to generates c code from an rti-file to " << std::endl;
  std::cerr << "1) register faudes-types and -functions with the run-time interface," << std::endl;
  std::cerr << "2) extract c declarations for bindings interface code." << std::endl;
  std::cerr << std::endl;
  std::cerr << "usage:" << std::endl;
  std::cerr << " rti2code [-flat] <rti input file> <output basename>" << std::endl;
  std::cerr << " rti2code -merge <rti input files> <output rti-file>" << std::endl;
  std::cerr << std::endl;
  std::cerr << "[note: the -flat option circumvents an issue with SWIG pre 4.1.0]" << std::endl;
  exit(1);
}


// ******************************************************************
// main programm
// ******************************************************************

int main(int argc, char *argv[]) {

  // Min args
  if(argc < 3) Usage();

  // Merge mode
  if(std::string(argv[1])=="-merge") {
    // Bail out
    if(argc < 4) Usage();
    // Load from files
    for(int i=2; i< argc-1; i++) {
      TypeRegistry::G()->MergeDocumentation(std::string(argv[i]));
      FunctionRegistry::G()->MergeDocumentation(std::string(argv[i]));
    }
    // Dump
    if(std::string(argv[argc-1]) != "-") {
      SaveRegistry(std::string(argv[argc-1]));
    } else {
      SaveRegistry();
    }
    return 0;
  } 

  // Test for "-flat" flag
  bool flat=false;
  if(std::string(argv[1])=="-flat") flat=true;
  if(flat &&  argc != 4) Usage();
  if(!flat &&  argc != 3) Usage();
  
  // Load registry file
  LoadRegistry(argv[argc-2]);

  // Code output streams
  std::ofstream rtiheader;
  rtiheader.open((std::string(argv[argc-1])+".h").c_str(), std::ios::out);
  std::ofstream rticode;
  rticode.open((std::string(argv[argc-1])+".cpp").c_str(), std::ios::out);
  std::ofstream luaheader;
  luaheader.open((std::string(argv[argc-1])+".i").c_str(), std::ios::out);

  // Introduce myself
  rtiheader << "/* rti2code: autogenerated libFAUDES rti registration: "; 
  rtiheader << VersionString() << " "  << PluginsString() << " */" << std::endl << std::endl; 
  rticode   << "/* rti2code: autogenerated libFAUDES rti registration: "; 
  rticode   << VersionString() << " "  << PluginsString() << " */" << std::endl << std::endl; 
  luaheader << "/* rti2code: autogenerated libFAUDES swig bindings declarations: "; 
  luaheader << VersionString() << " "  << PluginsString() << " */" << std::endl << std::endl; 

  // C++ static objects: auto load types
  rticode << "namespace faudes {" << std::endl;
  rticode << "/* Auto-register faudes types */" << std::endl;

  // Traverse type registry to figure faudes types
  TypeRegistry::Iterator  tit;
  int tcnt;
  for(tit=TypeRegistry::G()->Begin(), tcnt=1; tit!=TypeRegistry::G()->End();tit++,tcnt++) {
    // Get c/f type 
    std::string ctype=tit->second->CType();
    std::string ftype=tit->second->Name();
    // Bail out if no C type specified
    if(ctype=="") continue;
    // Remove name space faudes
    size_t pos=ctype.find("faudes::");
    if(pos!=std::string::npos) 
      ctype=ctype.substr(std::string("faudes::").length());
    // Bail out no auto-registration specified
    if(!tit->second->AutoRegistered()) continue;
    // report
    std::cout << "rti2code: generating auto-registration code for \"" << ftype << "\"" << std::endl;
    // Produce c code
    std::string rtiname = std::string("gRti") + ToStringInteger(tcnt) + "Register" + ftype;
    rticode << "AutoRegisterType<" << ctype << "> " << rtiname << "(\"" << ftype <<"\");";
    rticode << std::endl;
    // Extra data set: element tag
    if(tit->second->XElementTag()!="") {
      rtiname = std::string("gRti") + ToStringInteger(tcnt) + "XElementTag" + ftype;
      rticode << "AutoRegisterXElementTag<" << ctype << "> " << rtiname << "(\"" << ftype << 
        "\", \"" << tit->second->XElementTag() << "\");";
      rticode << std::endl;
    }
  }

  // C++ static objects: auto load types end
  rticode << "} // namespace" << std::endl;

  // C++ function declaration: load types
  rtiheader << "namespace faudes {" << std::endl;
  rtiheader << "void LoadRegisteredTypes(void);" << std::endl;
  rtiheader << "} // namespace" << std::endl;

  // C++ function definition: load types
  rticode << "namespace faudes {" << std::endl;
  rticode << "/* Register faudes types */" << std::endl;
  rticode << "void LoadRegisteredTypes(void) {" << std::endl;

  // Traverse type registry to figure faudes types
  for(tit=TypeRegistry::G()->Begin(); tit!=TypeRegistry::G()->End();tit++) {
    // Get C type 
    std::string ctype=tit->second->CType();
    // Bail out if no c type specified
    if(ctype=="") continue;
    // Remove name space faudes
    size_t pos=ctype.find("faudes::");
    if(pos!=std::string::npos) 
      ctype=ctype.substr(std::string("faudes::").length());
    // Report
    std::cout << "rti2code: generating registration code for \"" << tit->second->Name() << "\"" << std::endl;
    // Produce c code
    rticode << "  TypeRegistry::G()->Insert<" << ctype << ">(\"" << tit->second->Name() <<"\");";
    rticode << std::endl;
  }

  // C++ function definition: load types end
  rticode << "}" << std::endl;
  rticode << "} // namespace" << std::endl;


  // C++ function declaration: load functions
  rtiheader << "namespace faudes {" << std::endl;
  rtiheader << "void LoadRegisteredFunctions(void);" << std::endl;
  rtiheader << "} // namespace" << std::endl;

  // C++ function definition: load functions
  rticode << "namespace faudes {" << std::endl;
  rticode << "/* Register faudes functions */" << std::endl;
  rticode << "void LoadRegisteredFunctions(void) {" << std::endl;

  // C++ class definition: Function derivates 
  rtiheader << "namespace faudes {" << std::endl;

  // Traverse function registry: define rti functions
  int fcnt=0;
  FunctionRegistry::Iterator  fit;
  for(fit=FunctionRegistry::G()->Begin(); fit!=FunctionRegistry::G()->End();fit++, fcnt++) {
    // Current function definition
    const FunctionDefinition* fdef = fit->second;
    // Get C type and faudes function name
    std::string ctype=fdef->CType();
    std::string fname = fdef->Name();  
    // Bail out if no c type specified
    if(ctype=="") continue;
    // Remove name space faudes
    size_t pos=ctype.find("faudes::");
    if(pos!=std::string::npos) 
      ctype=ctype.substr(std::string("faudes::").length());
    // Bail out if no signature
    if(fdef->VariantsSize()==0) {
      std::cout << "rti2cocde: function registration: " << fname << ": no signatures" << std::endl;
      continue;
    }
    // Interpret signatures: set up type array
    std::vector< std::vector<std::string> > cparams;
    std::vector< std::vector<Parameter::ParamAttr> > cattrib;
    std::vector< std::vector<bool> > cretval;
    cparams.resize(fdef->VariantsSize());
    cattrib.resize(fdef->VariantsSize());
    cretval.resize(fdef->VariantsSize());
    // Loop all signatures
    for(int i=0; i<fdef->VariantsSize(); i++) {
      const Signature& sigi=fdef->Variant(i);
      int retcount=0;
      for(int j=0; j<sigi.Size(); j++) {
        // Retrieve faudes type and attrib
        std::string ftype=sigi.At(j).Type();
	Parameter::ParamAttr fattr=sigi.At(j).Attribute();
	bool fcret=sigi.At(j).CReturn();   
        // Count ret values
        if(fcret) retcount++;
        // Bail out on unknown faudestype
        if(!TypeRegistry::G()->Exists(ftype)) break;
        // Get corresponding ctype
        std::string ctype=TypeRegistry::G()->Definition(ftype).CType();
        // Bail out on unknown ctype
        if(ctype=="") break;
        // bail out on non-out ret value
        if(fcret && !(fattr==Parameter::Out)) break;
        // Bail out on undef attribute
        if(fattr==Parameter::UnDef) break;
        // Bail out on more than one ret values
        if(retcount>1) break;
        // Remove name space faudes
        size_t pos=ctype.find("faudes::");
        if(pos!=std::string::npos) 
          ctype=ctype.substr(std::string("faudes::").length());
        // Param ok
        cparams.at(i).push_back(ctype);
        cattrib.at(i).push_back(fattr);
        cretval.at(i).push_back(fcret);
      } 
      // Test for signature error
      if((int) cparams.at(i).size()!=sigi.Size()) {
        std::cout << "rti2code: function registration: " << fname << ": cannot interpret signature " 
		  << sigi.Name() << std::endl;
        cparams.resize(i);
        break;
      }
    }
    // Report
    std::cout << "rti2code: generating rti wrapper for \"" << fdef->Name() << "\"" << 
      " #" << cparams.size() << " variants" << std::endl;
    // Produce c code: register all functions function
    std::string rtiname = std::string("Rti") + ToStringInteger(fcnt) + ctype;
    rticode << "  FunctionRegistry::G()->Insert<" << rtiname << ">(\"" << fname <<"\");" << std::endl;
    // Produce c code: class declaration intro
    rtiheader << "/* Function class for C++ function " << ctype << "*/" << std::endl;
    rtiheader << "class " << rtiname << " : public Function { " << std::endl;
    rtiheader << "public:" << std::endl;
    rtiheader << "  using Function::operator=;" << std::endl;
    rtiheader << rtiname << "(const FunctionDefinition* fdef) : Function(fdef) {};" << std::endl;
    rtiheader << "virtual Function* New(void) const { return new " << rtiname << "(pFuncDef); };" << std::endl;
    rtiheader << "protected:" << std::endl;
    // Produce c code: function class: have typed param
    for(unsigned int i=0; i<cparams.size(); i++) 
      for(unsigned int j=0; j<cparams.at(i).size(); j++) 
        rtiheader << cparams.at(i).at(j) << "* " << "mP_" << i << "_" << j << ";" << std::endl;
    // Produce c code: function class: do type check
    rtiheader << "virtual bool DoTypeCheck(int n) {" << std::endl;
    rtiheader << "  bool res=false;" << std::endl;
    rtiheader << "  switch(mVariantIndex) { "<< std::endl;
    for(unsigned int i=0; i<cparams.size(); i++) {
      rtiheader << "  case " << i << ": { // variant " << fdef->Variant(i).Name() << std::endl;
      rtiheader << "    switch(n) { "<< std::endl;
      for(unsigned int j=0; j<cparams.at(i).size(); j++) {
        rtiheader << "    case " << j << ": ";
        rtiheader << "    res=DoTypeCast<" << cparams.at(i).at(j) << ">(" << j << ", mP_" << i <<"_" << j << "); ";
        rtiheader << "break; "<< std::endl;
      }
      rtiheader << "    default: break; " << std::endl;
      rtiheader << "    } "<< std::endl;
      rtiheader << "    break; "<< std::endl;
      rtiheader << "  } "<< std::endl;
    }
    rtiheader << "  default: break; " << std::endl;
    rtiheader << "  } "<< std::endl;
    rtiheader << "  return res;" << std::endl;
    rtiheader << "};" << std::endl;
    // Produce c code: function class: do execute
    rtiheader << "virtual void DoExecute(void) {" << std::endl;
    // Produce c code: do execute: switch variant
    rtiheader << "  switch(mVariantIndex) { "<< std::endl;
    for(unsigned int i=0; i<cparams.size(); i++) {
      rtiheader << "  case " << i << ": { // variant " << fdef->Variant(i).Name() << std::endl;
      rtiheader << "    ";
      // Figure return value (if any)
      for(unsigned int j=0; j<cparams.at(i).size(); j++) {
        if(!cretval.at(i).at(j)) continue;
        // Special case: integer
        if(cparams.at(i).at(j) == "Integer") {
          rtiheader << "*(mP_" << i << "_" << j << "->CReference()) = ";
        } else 
        // Special case: boolean
	  if(cparams.at(i).at(j) == "Boolean") {
          rtiheader << "*(mP_" << i << "_" << j << "->CReference()) = ";
        } else
        // Special case: integer
        if(cparams.at(i).at(j) == "String") {
          rtiheader << "*(mP_" << i << "_" << j << "->CReference()) = ";
        } else
        // Std case
	  rtiheader << "*mP_" << i << "_" << j  << " = ";
      }
      // Function name
      rtiheader << ctype <<"(";
      // Parameters
      int parpos=0;
      for(unsigned int j=0; j<cparams.at(i).size(); j++) {
        if(cretval.at(i).at(j)) continue;
        if((parpos++)!=0) rtiheader << " ,";
        // Special case: integer
        if(cparams.at(i).at(j) == "Integer") {
          rtiheader << "*(mP_" << i << "_" << j << "->CReference())";
        } else 
        // Special case: boolean
	  if(cparams.at(i).at(j) == "Boolean") {
          rtiheader << "*(mP_" << i << "_" << j << "->CReference())";
        } else
        // Special case: integer
        if(cparams.at(i).at(j) == "String") {
          rtiheader << "*(mP_" << i << "_" << j << "->CReference())";
        } else
        // Std case
          rtiheader << "*mP_" << i << "_" << j;
      }
      rtiheader << "); break; };" << std::endl;
    }
    // Produce c code: switch variant; done
    rtiheader << "  default: break; " << std::endl;
    rtiheader << "  }; "<< std::endl;
    // Produce c code: do execute: done
    rtiheader << "}; "<< std::endl;
    // Produce c code: function class: done
    rtiheader << "};"  << std::endl;

    // Produce swig code: lua function definition(s)    
    luaheader << "/* faudes-function \"" << fname << "\" */" << std::endl;
    // Figure my plugin to insert a conditional
    if(!flat) {
      std::string plugin=fdef->PlugIn();
      luaheader << "#if " << "( SwigModule == \"Swig" << plugin << "\")";
      //luaheader << " || ( SwigModule == \"SwigLibFaudes\")";  // requires SWIG 4.1
      luaheader << std::endl;
    }
    // Use C-type function name
    if(ctype!=fname)
      luaheader << "%rename(" << fname << ") " << ctype << ";" << std::endl;

    // Prepare swig code: preprocessed array
    std::vector< std::string > lfdefs;
    std::vector< std::string > lrtypes;
    std::vector< std::string > lhelp;
    // Prepare swig code: generate per signature
    for(unsigned int i=0; i<cparams.size(); i++) {
      // Create ctype function declaration: return value
      std::string lrtype="void";
      for(unsigned int j=0; j<cparams.at(i).size(); j++) {
        if(!cretval.at(i).at(j)) continue;
        // Special case: integer
        if(cparams.at(i).at(j) == "Integer") {
          lrtype="long int";
        } else 
        // Special case: boolean
	if(cparams.at(i).at(j) == "Boolean") {
          lrtype="bool";
        } else
        // Special case: string
        if(cparams.at(i).at(j) == "String") {
        lrtype="std::string";
        } else
        // Std case ctype as refernce
	lrtype = cparams.at(i).at(j);
        // No more than one return value
        break;
      }
      lrtypes.push_back(lrtype);
      // Create ctype function declaration: function body
      std::string lfdef = ctype + "(";
      // Create ctype function declaration: parameters
      int parpos=0;
      for(unsigned int j=0; j<cparams.at(i).size(); j++) {
        if(cretval.at(i).at(j)) continue;
        if((parpos++)!=0) lfdef += ", ";
        // Have const for +In+
	if(cattrib.at(i).at(j)==Parameter::In)
          lfdef +=  "const ";
        // Special case: integer
        if(cparams.at(i).at(j) == "Integer") {
          lfdef += "long int&";
        } else 
        // Special case: boolean
	if(cparams.at(i).at(j) == "Boolean") {
          lfdef += "bool&";
        } else
        // Special case: string
        if(cparams.at(i).at(j) == "String") {
          lfdef += "std::string&";
        } else
        // Std case ctype as refernce
  	  lfdef += cparams.at(i).at(j) + "&";
        // Mark elementary outputs
        if(cparams.at(i).at(j) == "Boolean" || cparams.at(i).at(j) == "String" 
           || cparams.at(i).at(j) == "Integer")
	if(cattrib.at(i).at(j)==Parameter::Out)
          lfdef += " OUTPUT";
      }
      // End of function declaration
      lfdef += ")"; 
      lfdefs.push_back(lfdef);
      // Add help entry: build nice signature
      std::string luasig = " " + fname + "(";
      bool leftcomma = false;
      bool rightcomma = false;
      for(unsigned int j=0; j<cparams.at(i).size(); j++) {
        // Special case: elementary output
        if(cparams.at(i).at(j) == "Boolean" || cparams.at(i).at(j) == "String" 
           || cparams.at(i).at(j) == "Integer")
	if(cattrib.at(i).at(j)==Parameter::Out) {
          if(leftcomma) luasig = "," + luasig;
	  //            if(leftcomma) luasig = ", " + luasig; // need tab in help system?
	  luasig=cparams.at(i).at(j) + luasig;
          leftcomma=true;
	  continue;
	}
        // Std case
        if(rightcomma) luasig += ", ";
        const Signature& sigi=fdef->Variant(i);
	luasig += sigi.At(j).Str();
        rightcomma=true;
      }  
      luasig+=")";
      // Add help entry: add with topic
      if(fdef->TextDoc()!=""){
        std::string topic= fdef->PlugIn();
        std::string key1=fdef->KeywordAt(1);
        if(topic=="CoreFaudes") {
          topic=fdef->KeywordAt(1);
          key1=fdef->KeywordAt(2);
        }
        if(topic.length()>0) topic.at(0)=toupper(topic.at(0));
        if(key1.length()>0) key1.at(0)=toupper(key1.at(0));
        lhelp.push_back("SwigHelpEntry(\"" + topic + "\", \"" + key1 + "\", \"" +
	  luasig + "\")");
      } else {
        lhelp.push_back("");
      }
    }
    // Filter pseudo doublets (only differ in lrtype)
    for(unsigned int i=1; i<lfdefs.size();i++) {
      unsigned int j;
      for(j=0; j<i; j++) 
	if(lfdefs.at(i)==lfdefs.at(j)) break;
      if(j==i) continue; 
      // Invalidate entry?
      if(lrtypes.at(j)=="void") 
        {lfdefs[j]=""; continue;}  
      if(lrtypes.at(i)=="void") 
        {lfdefs[i]=""; continue;}  
    } // Done: prepare per signature

    // Generate swig definitions: write
    int lcount=0;
    for(unsigned int i=0; i<lfdefs.size(); i++) {
      if(lfdefs.at(i)=="") continue;
      luaheader << lrtypes.at(i) << " " << lfdefs.at(i) << ";" << std::endl;
      lcount++;
      if(lhelp.at(i)=="") continue; 
      luaheader << lhelp.at(i) << ";" << std::endl;
    }
    std::cout << "rti2code: generating swig interface for function \"" << fdef->Name() << "\"" << 
      " #" << lcount << " variants" << std::endl;

    // End all signatures, incl conditional
    if(!flat) {
      luaheader << "#endif " << std::endl;
    }  
    luaheader << std::endl;


  } // Loop all functions

  // C++ class definition: function class: all such done
  rtiheader << "} // namespace" << std::endl;

  // C++ class definition: register function prototypes: done
  rticode << "}" << std::endl;
  rticode << "} // namespace" << std::endl;

  return(0);
}
