/** @file pd_alg_nb_sub_b_test.cpp  Unit Tests */


/* Pushdown plugin for FAU Discrete Event Systems Library (libfaudes)

   Copyright (C) 2013  Stefan Jacobi, Sven Schneider, Anne-Kathrin Hess

*/
#include "pd_alg_nb_sub_b_test.h"

namespace faudes {

/* *****************
 * TestRenqRenaming
 * *****************/
void TestRenQRenaming(){
  std::string name = "RenQ Renaming";
  std::cout << "Testing " << name << " ..." <<  std::endl;
  
  std::string renameString = "old";
  
  PushdownGenerator g1 = TestGenerator1();
  PushdownGenerator g2 = RenQ(renameString,g1);
  
  try{
    StateSet::Iterator its;    
    for(its = g2.StatesBegin(); its != g2.StatesEnd(); its++){
      
      const MergeStateAnnotation* msa = dynamic_cast<const MergeStateAnnotation*>(g2.StateAttribute(*its).Merge());
      
      if(msa == NULL)
        throw Exception(name, "MergeStateAnnotation not set.", 1003);
      if(msa->Annotation().compare(renameString) != 0){
        std::stringstream errstr;
        errstr << "Annotation incorrect, was " << msa->Annotation() << ", but " << renameString << " was expected." << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
  }
  catch (Exception e){ 
  }
  std::cout << "Finished " << name << std::endl;
}

/* *****************
 * TestRenQNumberOfStates
 * *****************/
void TestRenQNumberOfStates(){
  std::string name = "RenQ Number of States";
  std::cout << "Testing " << name << " ..." <<  std::endl;
  
  std::string renameString = "old";
  
  PushdownGenerator g1 = TestGenerator1();
  PushdownGenerator g2 = RenQ(renameString,g1);
  
  try{
    if(g1.Size() != g2.Size()){
        std::stringstream errstr;
        errstr << "Incorrect, number of states, was " << g1.Size() << ", but " << g2.Size() << " was expected." << std::endl;
        throw Exception(name, errstr.str(), 1003);
    } 
  }
  catch (Exception e){ 
  }
  std::cout << "Finished " << name << std::endl;
}

/* *****************
 * TestRenGRenamingStackSymbols
 * *****************/
void TestRenGRenamingStackSymbols(){
  std::string name = "RenG Renaming Stack Symbols";
  std::cout << "Testing " << name << " ..." <<std::endl;
  
  std::string renameString = "old";
  
  PushdownGenerator g1 = TestGenerator1();
  PushdownGenerator g2 = RenG(renameString,g1);
  
  try{
    StackSymbolSet::Iterator it;
    for(it = g1.StackSymbolsBegin(); it != g1.StackSymbolsEnd(); it++){
      //lambda does not get renamed so test for not getting renamed instead
      if(g2.IsStackSymbolLambda(*it)){
        if(!g2.StackSymbols().Exists(g2.StackSymbolName(*it))){
          std::stringstream errstr;
          errstr << "Annotation incorrect, " << g2.StackSymbolName(*it) << " was expected, but not found in the stack symbol list:\n" ;
          StackSymbolSet::Iterator iterr;
          for(iterr = g2.StackSymbolsBegin(); iterr != g2.StackSymbolsEnd(); iterr++){
            errstr << " " << g2.StackSymbolName(*iterr);
          }
          errstr << std::endl;
          throw Exception(name, errstr.str(), 1003);
        }
      }
      //all other symbols must get renamed
      else{
        if(!g2.StackSymbols().Exists(renameString + "-" + g2.StackSymbolName(*it))){
          std::stringstream errstr;
          errstr << "Annotation incorrect, " << renameString << "-" << g2.StackSymbolName(*it) << " was expected, but not found in the stack symbol list:\n" ;
          StackSymbolSet::Iterator iterr;
          for(iterr = g2.StackSymbolsBegin(); iterr != g2.StackSymbolsEnd(); iterr++){
            errstr << " " << g2.StackSymbolName(*iterr);
          }
          errstr << std::endl;
          throw Exception(name, errstr.str(), 1003);
      }
      }
  
    }
  }
  catch(Exception e){
  }
  std::cout << "Finished " << name << std::endl;
}

/* *****************
 * TestRenGNumberOfStackSymbols
 * *****************/
void TestRenGNumberOfStackSymbols(){
  std::string name = "RenG Number of Stack Symbols";
  std::cout << "Testing " << name << " ..." <<  std::endl;
  
  std::string renameString = "old";
  
  PushdownGenerator g1 = TestGenerator1();
  PushdownGenerator g2 = RenG(renameString,g1);
  
  try{
     if(g1.StackSymbols().Size() != g2.StackSymbols().Size()){
        std::stringstream errstr;
        errstr << "Incorrect number of stack symbols, was " << g1.StackSymbols().Size() << ", but " << g2.StackSymbols().Size() << " was expected." << std::endl;
        throw Exception(name, errstr.str(), 1003);
    } 
  }
  catch (Exception e){ 
  }
  std::cout << "Finished " << name << std::endl;
}

/* *****************
 * TestRenGRenamingTransitions
 * *****************/
void TestRenGRenamingTransitions(){
  
  FAUDES_TEST_DUMP("test 77",true)
  
  std::string name = "RenG Renaming Transitions";
  std::cout << "Testing " << name << " ..." <<  std::endl;
  
  std::string renameString = "old";
  
  PushdownGenerator g1 = TestGenerator1();
  PushdownGenerator g2 = RenG(renameString,g1);
  
  TransSet::Iterator tit;
  std::vector<Idx> oldPush, oldPop, push, pop;
  std::vector<Idx>::const_iterator pushit, popit;
  PopPushSet popPush;
  PopPushSet::const_iterator ppit;

  try{
    for(tit = g2.TransRelBegin(); tit != g2.TransRelEnd(); tit++){
      //examine all pop/push pairs
      popPush = g2.PopPush(*tit);
      for(ppit = popPush.begin(); ppit != popPush.end(); ppit++){
        
        //test pop stack symbols
        oldPop = ppit->first;
        for(popit = oldPop.begin(); popit != oldPop.end(); popit++){
          
          //everything but lambda must be renamed
          if(!g2.IsStackSymbolLambda(*popit)){
            std::string annotation = g2.StackSymbolName(*popit).substr(0, renameString.size() + 1);
            if(annotation.compare(renameString + "-") != 0){
              std::stringstream errstr;
              errstr << "Annotation incorrect, " << renameString << "- was expected, but symbol was " << g2.StackSymbolName(*popit) << std::endl;
              throw Exception(name, errstr.str(), 1003);
            }
          }
        }
        
        //test push stack symbols
        oldPush = ppit->second;
        for(pushit = oldPush.begin(); pushit != oldPush.end(); pushit++){
          
          //everything but lambda must be renamed
          if(!g2.IsStackSymbolLambda(*pushit)){
            std::string annotation = g2.StackSymbolName(*pushit).substr(0, renameString.size() + 1);
            if(annotation.compare(renameString + "-") != 0){
              std::stringstream errstr;
              errstr << "Annotation incorrect, " << renameString << "- was expected, but symbol was " << g2.StackSymbolName(*pushit) << std::endl;
              throw Exception(name, errstr.str(), 1003);
            }
          }
        }
      }
      
    }
  }
  catch(Exception e){
      FAUDES_TEST_DUMP("test 77 ERROR ",false)
  }
  std::cout << "Finished " << name << std::endl;
}

/* *****************
 * TestRep0NoLambdaPop
 * *****************/
void TestRep0NoLambdaPop(){
  std::string name = "Rep0 No Lambda Pop";
  std::cout << "Testing " << name << " ..." <<  std::endl;
  
  PushdownGenerator g1 = TestGenerator1();
  PushdownGenerator g2 = Rep0(g1);
  
  try{
    
    //iterate over all transitions
    TransSet::Iterator tit;
    PopPushSet::const_iterator ppsit;
    for(tit = g2.TransRel().Begin(); tit != g2.TransRel().End(); tit++){
      
      //get all PopPush pairs, extract pop (first), extract foremost
      //stack symbol (front), and test for lambda
      for(ppsit = g2.PopPush(*tit).begin(); ppsit != g2.PopPush(*tit).end(); ppsit++){
        if(g2.IsStackSymbolLambda(ppsit->first.front())){
          std::stringstream errstr;
          errstr << "Lambda popping edges not removed, Lambda pop found in transition" << g2.TransRel().Str(*tit) << std::endl;
          throw Exception(name, errstr.str(), 1003);
        }
      }
    }
  }
   catch(Exception e){
  }
  std::cout << "Finished " << name << std::endl;
}

/* *****************
 * TestRep0AllExpectingTransition
 * *****************/
void TestRep0AllExpectingTransition(){
  std::string name = "Rep0 All Expecting Transition";
  std::cout << "Testing " << name << " ..." <<  std::endl;

  PushdownGenerator g1 = TestGenerator1();
  PushdownGenerator g2 = Rep0(g1);
 
  //test g1 for lambda transition
  TransSet::Iterator tit;
  PopPushSet::const_iterator ppsit;
  bool hasLambda;
  for(tit = g1.TransRel().Begin(); tit != g1.TransRel().End(); tit++){
    
    //get all PopPush pairs, extract pop (first), extract foremost
    //stack symbol (front), and test for lambda
    for(ppsit = g1.PopPush(*tit).begin(); ppsit != g1.PopPush(*tit).end(); ppsit++){
      if(g2.IsStackSymbolLambda(ppsit->first.front())){
        hasLambda = true;
        //lambda was found, no need to search further
        break;
      }
    }
  }
  
  if(!hasLambda){
      std::cout << "warning: cannot perform test, because there is no lambda pop in the original generator" << std::endl;
      return;
  }
  
  //test g2 for transition (s1,ev,s2,u,uw), where u is every stack symbol
  //except for lambda and bottom
  try{
    
    //iterate over all transitions
    std::vector<Idx> pop, push;
    std::vector<Idx>::const_iterator popit, pushit;
    StackSymbolSet::Iterator ssit;
    bool transFound, symbolFound;
    for(tit = g2.TransRel().Begin(); tit != g2.TransRel().End(); tit++){
      transFound = true;
      
      //iterate over all relevant stack symbols and see if there is one PopPush
      //pair with pop u and push wu
      for(ssit = g2.StackSymbols().Begin(); ssit != g2.StackSymbols().End(); ssit++){

        //lambda and stack bottom are not relevant
        if (*ssit == g2.StackBottom() || g2.IsStackSymbolLambda(*ssit)) continue;
        
        //test PopPush pairs and find at least one (u,uw) PopPush pair per symbol
        symbolFound = false;
        for(ppsit = g2.PopPush(*tit).begin(); ppsit != g2.PopPush(*tit).end(); ppsit++){

          pop = ppsit->first;
          push = ppsit->second;
          //if front of pop or push are  identical, the pair is found
          if(pop.front() == *ssit && *push.rbegin() == *ssit){
            symbolFound = true;
            break;
          }
        }
        //if any one symbol has not been found, this transition is not relevant
        if(!symbolFound){
          transFound = false;
          break;
        }
      }
      //no need to look for another such transition
      if(transFound) break;
    }
    
    if(!transFound){
      std::stringstream errstr;
      errstr << "Original generator has lambda popping edge, but result generator has no edge that accepts all stack symbols" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
   catch(Exception e){
  }
  std::cout << "Finished " << name << std::endl;
}

/* ****************
 * TestRep2NumberOfTransitions
 * *****************/
void TestRep2NumberOfStatesTransitions(){
  
  std::string name = "Rep2 Number of States and Transitions";
  TestStart(name);
  
  PushdownGenerator g1 = TestGenerator4();
  
  //expected resulst for this particular test generator
  Idx expectedNumberTransitions = 7;
  Idx expectedNumberStates = 6;
  
  PushdownGenerator g2 = Rep2(g1);
  
  try{
      
    //test for number of states
    if(g2.Size() != expectedNumberStates){
      std::stringstream errstr;
      errstr << "Number of states  was " << g2.Size() << ", but " << expectedNumberStates << " was expected." << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    
    //test for number of transitions
    if(g2.TransRelSize() != expectedNumberTransitions){
      std::stringstream errstr;
      errstr << "Number of transitions incorrect" << g2.TransRelSize() << ", but " << expectedNumberTransitions << " was expected." << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
  catch (Exception e){ 
  }
  
  TestEnd(name);
}

/* *****************
 * TestRep2Renaming
 * *****************/
void TestRep2Renaming(){
  std::string name = "Rep2 Renaming";
  TestStart(name);
  
  std::string renameString = "old";
  
  PushdownGenerator g1 = TestGenerator4();
  
  PushdownGenerator g2 = Rep2(g1);
  
  try{
    StateSet::Iterator stateit;    
    for(stateit = g2.StatesBegin(); stateit != g2.StatesEnd(); stateit++){
      
      const MergeStateAnnotation* msa = dynamic_cast<const MergeStateAnnotation*>(g2.StateAttribute(*stateit).Merge());
      
      if(msa == NULL)
        throw Exception(name, "MergeStateAnnotation not set.", 1003);
      if(msa->Annotation().compare(renameString) != 0){
        std::stringstream errstr;
        errstr << "Annotation incorrect, was " << msa->Annotation() << ", but " << renameString << " was expected." << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
  }
  catch (Exception e){ 
  }
  TestEnd(name);
}

/* ****************
 * TestRppReadPopPushOnly
 * *****************/
void TestRppNumberStatesTransitions(){
  //TODO dont know what to expect
//   std::string name = "Rpp Number States Transitions";
//   TestStart(name);
//   
//   PushdownGenerator g1 = TestGenerator6();
//   
//   PushdownGenerator g2 = Rpp(g1);
//   try{
//     //number of states must be 7
//     if(g2.Size() != 7){
//       std::stringstream errstr;
//       errstr << "Number of states was expected to be 7, but was " << g2.Size() << "." << std::endl;
//       throw Exception(name, errstr.str(), 1003);
//     }
//     
//     TransSet::Iterator transit;    
//     PopPushSet::const_iterator ppit;
//     //number of transitions must be 10
//     int transCount = 0;
//     for(transit = g2.TransRelBegin(); transit != g2.TransRelEnd(); transit++){
//       transCount += g2.PopPush(*transit).size();
//     }
//     
//     if(transCount != 10){
//       std::stringstream errstr;
//       errstr << "Number of transitions was expected to be 10, but was " << transCount << "." << std::endl;
//       throw Exception(name, errstr.str(), 1003);
//     }
//   }
//   catch (Exception e){ 
//   }
//   TestEnd(name);
}

/* ****************
 * TestRppReadPopPushOnly
 * *****************/
void TestRppReadPopPushOnly(){
  std::string name = "Rpp Read Pop Push Only";
  TestStart(name);
  
  PushdownGenerator g1 = TestGenerator6();
  
  PushdownGenerator g2 = Rpp(g1);
  try{
    TransSet::Iterator transit;    
    PopPushSet::const_iterator ppit;
    std::vector<Idx> pop, push;
    std::vector<Idx>::const_iterator ssit;
    for(transit = g2.TransRelBegin(); transit != g2.TransRelEnd(); transit++){
      for(ppit = g2.PopPushBegin(*transit); ppit != g2.PopPushEnd(*transit); ppit++){
        
        //read pop and push for convenience
        pop = ppit->first;
        push = ppit->second;
        
        //check for read only transition
        if(!g2.IsEventLambda(transit->Ev) &&
          pop == push){
          continue;
        }
        //check for pop only transition
        else if(g2.IsEventLambda(transit->Ev) &&
          !g2.IsStackSymbolLambda(pop.front()) &&
          pop.size() == 1 &&
          g2.IsStackSymbolLambda(push.front())){
          continue;
        }
        //check for push only transition
        else if(g2.IsEventLambda(transit->Ev) &&
          push.size() == 2 &&
          pop.size() == 1 &&
          pop.front() == push.back()){
          continue;
        }
        //error
        else{
          std::stringstream errstr;
          errstr << "Transition (" << transit->X1 << ", " << g2.EventName(transit->Ev) << ", " << transit->X2 << ") with pop [";
          for(ssit = pop.begin(); ssit != pop.end(); ssit++){
            errstr << " " << g2.StackSymbolName(*ssit);
          }
          errstr << "] and push [";
          for(ssit = push.begin(); ssit != push.end(); ssit++){
            errstr << " " << g2.StackSymbolName(*ssit);
          }
          errstr << "] was neither read nor pop nor push." << std::endl;
          throw Exception(name, errstr.str(), 1003);
        }
      }
    }
  }
  catch (Exception e){ 
  }
  TestEnd(name);
}

/* ****************
 * TestNdaActivePassive
 * *****************/
void TestNdaActivePassive(){
  std::string name = "Nda Active Passive";
  TestStart(name);
  
  PushdownGenerator g1 = TestGenerator7();
  PushdownGenerator g2 = Nda(g1);
  
  try{
    //the number of transitions must have doubled
    if(2*g1.States().Size() != g2.States().Size()){
      std::stringstream errstr;
      errstr << "Number of states incorrect, was" << g2.States().Size() << ", but " << 2*g1.States().Size() << " was expected." << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    
    StateSet::Iterator stateit;
    int active = 0;
    int passive = 0;
    //there must be an equal amount of active and passive states
    for(stateit = g2.StatesBegin(); stateit != g2.StatesEnd(); stateit++){
      
      const MergeStateAnnotation* msa = dynamic_cast<const MergeStateAnnotation*>(g2.StateAttribute(*stateit).Merge());
      
      if(msa == NULL)
        throw Exception(name, "MergeStateAnnotation not set.", 1003);
      
      if(msa->Annotation().compare("active") == 0){
        active++;
      }
      else if(msa->Annotation().compare("passive") == 0){
        passive++;
      }
      else{
        std::stringstream errstr;
        errstr << "Annotation incorrect, was " << msa->Annotation() << ", but either active or passive was expected." << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
    if(active != passive){
      std::stringstream errstr;
      errstr << "There were " <<  active << " active states and " << passive << " passive states, but equal numbers were expected." << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
  catch (Exception e){ 
  }
  
  TestEnd(name);
}

/* ****************
 * TestNdaTransitions
 * *****************/
void TestNdaTransitions(){
  std::string name = "Nda Transitions";
  TestStart(name);
  
  PushdownGenerator g1 = TestGenerator7();
  PushdownGenerator g2 = Nda(g1);
  
  //put together expected transitions for later comparison
  std::set<std::pair<Idx,Idx> > expectedTransitions;
  expectedTransitions.insert(std::make_pair(1,3));
  expectedTransitions.insert(std::make_pair(1,5));
  expectedTransitions.insert(std::make_pair(2,3));
  expectedTransitions.insert(std::make_pair(2,6));
  expectedTransitions.insert(std::make_pair(3,8));
  expectedTransitions.insert(std::make_pair(4,8));
  expectedTransitions.insert(std::make_pair(5,3));
  expectedTransitions.insert(std::make_pair(6,4));
  expectedTransitions.insert(std::make_pair(7,6));
  expectedTransitions.insert(std::make_pair(8,6));
  
  try{
    TransSet::Iterator transit;
    for(transit = g2.TransRelBegin(); transit != g2.TransRelEnd(); transit++){

      if(expectedTransitions.erase(std::make_pair(transit->X1, transit->X2)) == 0){
        std::stringstream errstr;
        errstr << "Transition from state " << transit->X1 << " to state " << transit->X2 << " found, but was not expected." << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
    
    if(expectedTransitions.size() != 0){
      std::stringstream errstr;
      errstr << "Not all expected transitions were found." << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
  catch (Exception e){ 
  }
  
  TestEnd(name);
}

/* *****************
 * TestRenQ
 * *****************/
void TestRenQ(){
  TestRenQRenaming();
  //TestRenQNumberOfStates();
}

/* *****************
 * TestRenG
 * *****************/
void TestRenG(){
  TestRenGRenamingStackSymbols();  
  TestRenGNumberOfStackSymbols();
  TestRenGRenamingTransitions();
}

/* *****************
 * TestRep0
 * *****************/
void TestRep0(){
  TestRep0NoLambdaPop();
  TestRep0AllExpectingTransition();
}

/* ****************
 * TestRpp
 * *****************/
void TestRpp(){
  TestRppNumberStatesTransitions();
  TestRppReadPopPushOnly();
}

/* *****************
 * TestRep2
 * *****************/
void TestRep2(){
  
  TestRep2NumberOfStatesTransitions();
  TestRep2Renaming();
}

/* *****************
 * TestNda
 * *****************/
void TestNda(){
  
  TestNdaActivePassive();
  TestNdaTransitions();
}

} // namespace faudes

