/** @file pd_alg_lrp.h  functions related to parsers*/


/* Pushdown plugin for FAU Discrete Event Systems Library (libfaudes)

   Copyright (C) 2013  Stefan Jacobi, Sven Schneider, Anne-Kathrin Hess

*/

#ifndef FAUDES_PD_ALG_LRP_H
#define FAUDES_PD_ALG_LRP_H

#include "corefaudes.h"
#include "pd_gotogenerator.h"
#include "pd_parser.h"
#include "pd_grammar.h"
#include "pd_alg_first.h"
 
namespace faudes {

/**
 * Find the successor states of a state for a given input symbol.
 * 
 * @param gotoGen
 *      the generator on which to work
 * @param state
 *      the state of which to find the successor state
 * @param symbol
 *      the input symbol
 * @return
 *      A set with the successor states. The set size is 1 if a successor state
 * is found. Set size is 0 if no successor state is found.
 */
std::set<Idx> GeneratorGoto(const GotoGenerator& gotoGen, Idx state, const GrammarSymbolPtr& symbol);

/**
 * Find the successor state sequence of a state for a given input word. If the word
 * cannot be completely matched to a state sequence, the state sequence will be
 * partially matched (i. e., it will be as long as the part of the word that can be
 * matched).
 * 
 * @param gotoGen
 *      the generator on which to work
 * @param state
 *      the state of which to find the successor state
 * @param word
 *      the input word
 * @return
 *      A set with the successor states. The set size will be anywhere between 0 and 
 * the length of the word.
 */
std::vector<Idx> GeneratorGotoSeq(const GotoGenerator& gotoGen, Idx startState, const GrammarSymbolVector& word);

/**
 * construct the parsing rules for shifting from a grammar and its LR(k) machine
 * 
 * @param gr
 *      the grammar on which to work
 * @param augGr
 *      augmented version of the grammar on which to work
 * @param gotoGen
 *      LR(k) machine of the grammar
 * @param k
 *      a natural number that denotes the kind of parser the function is working
 * on (LR(k) parser) (will currently only work for LR(1))
 * @return
 *      set of shift actions
 */
std::set<Lr1ParserAction> LrpShiftRules(const Grammar& gr, const Grammar& augGr, const GotoGenerator& gotoGen, uint k);

/**
 * construct the parsing rules for reducing from a grammar and its LR(k) machine
 * 
 * @param gr
 *      the grammar on which to work
 * @param augGr
 *      augmented version of the grammar on which to work
 * @param gotoGen
 *      LR(k) machine of the grammar
 * @param k
 *      a natural number that denotes the kind of parser the function is working
 * on (LR(k) parser) (will currently only work for LR(1))
 * @return
 *      set of reduce actions
 */
std::set<Lr1ParserAction> LrpReduceRules(const Grammar& gr, const Grammar& augGr, const GotoGenerator& gotoGen, uint k);

/**
 * construct all parsing rules from a grammar and its LR(k) machine
 * 
 * @param gr
 *      the grammar on which to work
 * @param augGr
 *      augmented version of the grammar on which to work
 * @param gotoGen
 *      LR(k) machine of the grammar
 * @param k
 *      a natural number that denotes the kind of parser the function is working
 * on (LR(k) parser) (will currently only work for LR(1))
 * @return
 *      set of parsing rules
 */
std::set<Lr1ParserAction> LrpRules(const Grammar& gr, const Grammar& augGr, const GotoGenerator& gotoGen, uint k);

/**
 * construct an LR(k) parser from a grammar and its LR(k) machine
 * 
 * @param gr
 *      the grammar on which to work
 * @param augGr
 *      augmented version of the grammar on which to work
 * @param gotoGen
 *      LR(k) machine of the grammar
 * @param k
 *      a natural number that denotes the kind of parser the function is working
 * on (LR(k) parser) (will currently only work for LR(1))
 * @param augSymbol
 *      the terminal with which the grammar was augmented to create the augmented
 * grammar
 * @return
 *      set of parsing rules
 */
Lr1Parser Lrp(const Grammar& gr, const Grammar& augGr, const GotoGenerator& gotoGen, uint k, const Terminal& augSymbol);

/**
 * Converts a parser into a pushdown generator. The stack symbols will be the parser's
 * nonterminals
 * 
 * @param gotoGen
 *      Lr(k) machine of the grammar that the parser parses
 * @param parser
 *      the parser to convert
 * @return
 *      the parser as a pushdown generator
 */
PushdownGenerator Gp2Pp(const GotoGenerator& gotoGen, const Lr1Parser& parser);

/**
 * Remove the augmentation from the generator (diminish it). This will remove all
 * transitions whose event is the augment symbol and make those transitions start
 * states final. It will also remove the augment symbol as an event.
 * 
 * @param pd
 *      generator to diminish
 * @param augSymbol
 *      the symbol with wich the grammar that was used to create the generator
 * was augmented
 * @return
 *      diminished generator
 */
PushdownGenerator Dim(const PushdownGenerator& pd, const Terminal& augSymbol);



} // namespace faudes

#endif
