/** @file pd_alg_cc.cpp  Top-level functions*/


/* Pushdown plugin for FAU Discrete Event Systems Library (libfaudes)

   Copyright (C) 2013  Stefan Jacobi, Sven Schneider, Anne-Kathrin Hess

*/

#include "pd_alg_cc.h"

namespace faudes {

  
/* *************************
 * Transient
 * *************************/
StateSet Transient(const PushdownGenerator& pd){
  
  TransSet::Iterator transit;
  StateSet rStates;
  //iterate over all transitions
  for(transit = pd.TransRelBegin(); transit != pd.TransRelEnd(); transit++){
    
    //add start state of lambda read transition to set
    if(pd.IsEventLambda(transit->Ev)){
      rStates.Insert(transit->X1);
    }
  }
  
  return rStates;  
}

/* *************************
 * Rnce
 * *************************/
PushdownGenerator Rnce(const PushdownGenerator& pd, const System& s){
  
  StateSet transientStates = Transient(pd);
  StateSet ucEars;
  EventSet uc = pd.UncontrollableEvents();
  StateSet::Iterator stateit;
  EventSet::Iterator eventit;
  Idx oldState;
  
  //iterate over all states
  for(stateit = pd.StatesBegin(); stateit != pd.StatesEnd(); stateit++){
    
    //get the merge attribute, which must must be either an ear or a heads
    const MergeStateSplit* mss = dynamic_cast<const MergeStateSplit*>(pd.Merge(*stateit));
    if(mss == NULL){
      std::stringstream errstr;
      errstr << "State in Rnce was neither head nor ear, check correctness of Merge attribute" << std::endl;
      throw Exception("Rnce", errstr.str(), 1001);
    }
    
    //only look at ears
    if(mss->IsHead()){
      continue;
    }
    
    //only look at transient ears
    if(transientStates.Exists(*stateit)){
      continue;
    }
    
    //check the current state for each uncontrollable event
    for(eventit = uc.Begin(); eventit != uc.End(); eventit++){
      
      //only look at states whose corresponding dfa state allows this 
      //uncontrollable event
      oldState = pd.DfaState(*stateit);
      if(!s.ExistsTransition(oldState,*eventit)){
        continue;
      }
            
      //only look at states that do not allow the uncontrollable event to happen
      if(pd.ExistsTransition(*stateit,*eventit)){
        continue;
      }
      
      //all these states are uncontrollable ears
      ucEars.Insert(*stateit);
    }
  } 
  PushdownGenerator rPd = pd;
  rPd.DelStates(ucEars);
  return rPd;
}

/* *************************
 * Times
 * *************************/
PushdownGenerator Times(const System& s, const PushdownGenerator& pd){
  
  //result generator
  PushdownGenerator rPd;
  
  //map to remember which states get merged into which
  std::map<std::pair<Idx,Idx>, Idx > stateMap;
  
  StateSet::Iterator pdStateit, sStateit;
  Idx newState;
  std::vector<Idx> oldStatesVector;
  //construct states as cartesian prodcut of both generators states
  for(pdStateit = pd.StatesBegin(); pdStateit != pd.StatesEnd(); pdStateit++){
    for(sStateit = s.StatesBegin(); sStateit != s.StatesEnd(); sStateit++){
      
      //add new state for each pair of states and save it in the map
      newState = rPd.InsState();
      stateMap.insert(std::make_pair(std::make_pair(*sStateit, *pdStateit), newState));
      
      //add information about parent states
      oldStatesVector.clear();
      oldStatesVector.push_back(*sStateit);
      oldStatesVector.push_back(*pdStateit);
      MergeStates oldStates = MergeStates(oldStatesVector);
      rPd.SetMerge(newState,oldStates);
      rPd.SetDfaState(newState,*sStateit);
      
      //if both states are init states, the new state is an init state as well
      if(*pdStateit == pd.InitState() && *sStateit == s.InitState()){
        rPd.SetInitState(newState);
      }
      
      //if both states are final states, the new state is a final state as well
      if(pd.ExistsMarkedState(*pdStateit) && s.ExistsMarkedState(*sStateit)){
        rPd.SetMarkedState(newState);
      }
    } 
  }

  //stack symbols are the same as the pushdown generator's stack symbols
  rPd.InsStackSymbols(pd.StackSymbols());

  //stack bottom is the same as the pushdown generator's stack bottom
  rPd.SetStackBottom(pd.StackSymbolObj(pd.StackBottom()));
  
  //the new event set is the intersection of both old event sets
  IntersectEvents(s,pd,rPd);

  //add lambda event if there was a lambda event in the pushdown generator's event set
  if(pd.Alphabet().Exists(FAUDES_PD_LAMBDA)){
    rPd.InsEvent(FAUDES_PD_LAMBDA);
  }
  
  Idx newX1, newX2;
  std::string sEvName, pdEvName;
  //add transitions to rPd. Iterate over all pushdown generator transitions
  TransSet::Iterator pdTransit, sTransit;
  PopPushSet::const_iterator ppsit;
  for(pdTransit = pd.TransRelBegin(); pdTransit != pd.TransRelEnd(); pdTransit++){
    
    //look at every pop push set of the transition
    for(ppsit = pd.PopPushBegin(*pdTransit); ppsit != pd.PopPushEnd(*pdTransit); ppsit++){
      
      pdEvName = pd.EventName(pdTransit->Ev);
      //for every no-read transition (q,lambda,w,w',q') of pd and every state p of s,
      //add a transition ((p,q),lambda,w,w',(p,q'))
      if(pdEvName.compare(FAUDES_PD_LAMBDA) == 0){
        for(sStateit = s.StatesBegin(); sStateit != s.StatesEnd(); sStateit++){
          
          //find new states corresponding to (p,q), and (p,q')
          newX1 = stateMap.find(std::make_pair(*sStateit, pdTransit->X1))->second;
          newX2 = stateMap.find(std::make_pair(*sStateit, pdTransit->X2))->second;
          
          //add transition
          rPd.SetTransition(newX1, rPd.EventIndex(FAUDES_PD_LAMBDA), newX2, ppsit->first, ppsit->second);
        }
      }
      //for every read transition (q,a,w,w',q') of pd and every transition (p,a,p')
      //of s, add a transition ((p,q),a,w,w',(p',q'))
      else{
        
        //iterate over all system transitions
        for(sTransit = s.TransRelBegin(); sTransit != s.TransRelEnd(); sTransit++){
          
          sEvName = s.EventName(sTransit->Ev);
          //compare the transition events
          if(pdEvName.compare(sEvName) == 0){
            
            //find new states corresponding to (p,q), and (p',q')
            newX1 = stateMap.find(std::make_pair(sTransit->X1, pdTransit->X1))->second;
            newX2 = stateMap.find(std::make_pair(sTransit->X2, pdTransit->X2))->second;
            
            //add transition
            rPd.SetTransition(newX1, rPd.EventIndex(sEvName), newX2, ppsit->first, ppsit->second);
          }
        }
      }
    }
  }
  return rPd;
}


/* *************************
 * IntersectEvents
 * *************************/
void IntersectEvents(const System& s, const PushdownGenerator& pd, PushdownGenerator& rPd){
  EventSet::Iterator pdEventit, sEventit;
  for(pdEventit = pd.AlphabetBegin(); pdEventit != pd.AlphabetEnd(); pdEventit++){
    for(sEventit = s.AlphabetBegin(); sEventit != s.AlphabetEnd(); sEventit++){
      if(*pdEventit == *sEventit){
        //controllable and observable
        if(pd.Controllable(*pdEventit) && 
          s.Controllable(*sEventit) && 
          pd.Observable(*pdEventit) &&
          s.Observable(*sEventit))
        {
          rPd.InsControllableEvent(*pdEventit);
        }
        //controllable and not observable
        else if(pd.Controllable(*pdEventit) &&
          s.Controllable(*sEventit) &&
          !pd.Observable(*pdEventit) &&
          !s.Observable(*sEventit))
        {
          rPd.InsControllableEvent(*pdEventit);
          rPd.ClrObservable(*pdEventit);
        }
        //not controllable, but observable
        else if(!pd.Controllable(*pdEventit) &&
          !s.Controllable(*sEventit) &&
          pd.Observable(*pdEventit) &&
          s.Observable(*sEventit))
        {
          rPd.InsEvent(*pdEventit);
        }
        //neither controllable nor observable
        else if(!pd.Controllable(*pdEventit) &&
          !s.Controllable(*sEventit) &&
          !pd.Observable(*pdEventit) &&
          !s.Observable(*sEventit))
        {
          rPd.InsUnobservableEvent(*pdEventit);
        }
      }
    }
  }
}

/* *************************
 * Split
 * *************************/
PushdownGenerator Split(const PushdownGenerator& pd){
  
  //copy the generator
  PushdownGenerator rPd = pd;
  
  //insert lambda event in case it has not already been added
  rPd.InsEvent(FAUDES_PD_LAMBDA);
  
  //clear transition relation and states, because they will be replaced
  rPd.ClearTransRel();
  StateSet::Iterator stateit=rPd.StatesBegin();
  while(stateit!=rPd.StatesEnd()){
    rPd.DelState(*(stateit++)); // fixed: 2013-12-17
  }
    
  //map ears and stack symbols to their heads for later reference
  std::multimap<Idx,std::pair<Idx,Idx> > headEarMap;
  //map ears and stack symbols to the states of the old generator for later reference.
  //headMap<headIdx,oldIdx>, earMap<oldIdx,headIdx>
  std::map<Idx,Idx> headMap, earMap;
  
  StackSymbolSet::Iterator ssit;
  Idx newHead, newEar;
  //insert new states as ears and heads
  for(stateit = pd.StatesBegin(); stateit != pd.StatesEnd(); stateit++){
    
    //insert a head for each state in the old generator
    newHead = rPd.InsState();
    MergeStateSplit mss(*stateit);
    rPd.SetMerge(newHead, mss);
    rPd.SetDfaState(newHead,pd.DfaState(*stateit));
    
    //if the state was the initial state in the old generator, this head
    //is an initial state as well
    if(pd.ExistsInitState(*stateit)){
      rPd.SetInitState(newHead);
    }
    
    //save head in map
    headMap.insert(std::make_pair(*stateit,newHead));
        
    //insert an ear for each state in the old generator and each stack symbol
    //except lambda
    for(ssit = rPd.StackSymbolsBegin(); ssit != rPd.StackSymbolsEnd(); ssit++){
      if(!rPd.IsStackSymbolLambda(*ssit)){
        newEar = rPd.InsState();
        MergeStateSplit mss(*stateit, *ssit);
        rPd.SetMerge(newEar, mss);
        rPd.SetDfaState(newEar,pd.DfaState(*stateit));
        
        //if the state was a final state in the old generator, this ears
        //is a final state as well
        if(pd.ExistsMarkedState(*stateit)){
          rPd.SetMarkedState(newEar);
        }
        
        //save ear in map
        earMap.insert(std::make_pair(newEar, *stateit));
        //save pair of head and ear in map
        headEarMap.insert(std::make_pair(newHead,std::make_pair(newEar,*ssit)));
        
      }
    }
  }
  
  TransSet::Iterator transit;
  PopPushSet::const_iterator popPushit;
  std::multimap<Idx,std::pair<Idx,Idx> >::iterator headEarit;
  std::vector<Idx> ssvector;
  Idx head, ear, startState, endState, event;
  Idx stackSymbol;
  //insert transitions
  //iterate over all ears
  for(headEarit = headEarMap.begin(); headEarit != headEarMap.end(); headEarit++){
    
    //insert transition from head to ear
    head = headEarit->first;
    ear = headEarit->second.first;
    stackSymbol = headEarit->second.second;
    
    ssvector.clear();
    ssvector.push_back(stackSymbol);
    
    rPd.SetTransition(head, rPd.EventIndex(FAUDES_PD_LAMBDA), ear, ssvector, ssvector);
    
    //connect ears to heads
    for(transit = pd.TransRelBegin(); transit != pd.TransRelEnd(); transit++){
      for(popPushit = pd.PopPushBegin(*transit); popPushit != pd.PopPushEnd(*transit); popPushit++){
        
        //determine if this ear is associated with the transitions start state and
        //determine if the ear has the same stack symbol as the transition's
        //pop symbol (the pop vector is supposed to contain only one stack symbol).
        //if both are true:
        if(earMap.find(ear)->second == transit->X1 && stackSymbol == *popPushit->first.begin()){
          
          //look up the head associated with the transition's end state
          endState = headMap.find(transit->X2)->second;
          startState = ear;
          event = rPd.EventIndex(pd.EventName(transit->Ev));
          
          //insert transition from ear to head
          rPd.SetTransition(startState, event, endState, popPushit->first, popPushit->second);
        }
      }
    } 
  }
  return rPd;
}

/* *************************
 * Nonblock
 * *************************/
void PushdownNonblock(const PushdownGenerator& pd, PushdownGenerator& res, bool print){
  
  //make DFA-coaccessible

  PushdownGenerator rPd = pd;
  rPd.Coaccessible();
   
  //restrict pd to read, pop and push rules
  if(print){
    rPd = Rpp(rPd);
    std::cout << "*******************" << std::endl;
    std::cout << "Nonblock rpp" << std::endl;
    std::cout << "*******************" << std::endl;
    rPd.Write();
  }
  else{
    rPd = Rpp(rPd);
  }
  
  
  //remove useless transitions created by rpp
  rPd = RemoveUselessTransitions(rPd);
 
  
  //remove double acceptance
  if(print){
    rPd = Nda(rPd);
    std::cout << "*******************" << std::endl;
    std::cout << "Nonblock nda" << std::endl;
    std::cout << "*******************" << std::endl;
    rPd.Write();
  }
  else{
    rPd = Nda(rPd);
  }
  
  rPd.Accessible();
  rPd.Coaccessible();
  rPd = TrimStackSymbols(rPd);
  
  //convert to grammar
  Grammar gr;
  if(print){
    gr = Sp2Lr2(rPd);
    std::cout << "*******************" << std::endl;
    std::cout << "Nonblock sp2lr" << std::endl;
    std::cout << "*******************" << std::endl;
    std::cout << gr.GrammarProductions().size() << std::endl;
    std::cout << gr.Str() << std::endl;
  }
  else{
    gr = Sp2Lr2(rPd);
  }
  
  //remove nonproductive productions
  gr = Rnpp(gr);
    
  //remove unreachable productions
  if(print){
    gr = Rup(gr);
    std::cout << "*******************" << std::endl;
    std::cout << "Nonblock rup" << std::endl;
    std::cout << "*******************" << std::endl;
    std::cout << gr.GrammarProductions().size() << std::endl;
    std::cout << gr.Str() << std::endl;
  }
  else{
    gr = Rup(gr);
  }
  
  //augment grammar 
  //create temporary automaton to create a new event for the augment terminal
  //and a new stack symbol for the augment nonterminal
  PushdownGenerator tempPd = rPd;
  //get fresh terminal
  Idx freshEventIdx;
  std::string freshEventName = "augSymbol";
  std::stringstream currentTry;
  currentTry << freshEventName;
  bool freshFound = false;
  int tries = 0;
  //try to get an unused event name to create a new terminal from
  while(!freshFound){
    if(tempPd.ExistsEvent(currentTry.str())){
      currentTry.clear();
      currentTry << freshEventName << tries;
      tries++;
      continue;
    }
    freshFound = true;
    freshEventIdx = tempPd.InsEvent(currentTry.str());
    
  }
  //create fresh terminal 
  Terminal t(freshEventIdx);  
 

  std::vector<Idx> v;
  //no need for this stack symbol to be unique, because no nonterminal should contain 0
  Idx stackSymbolIdx = tempPd.InsStackSymbol(StackSymbol("augSymbol"));
  v.push_back(stackSymbolIdx);
  Nonterminal nt(0,v);
  
  Grammar augGr = Aug(gr, nt, t);
  
  //construct the goto generator for the augmented grammar
  GotoGenerator gotoGen;
  if(print){
    gotoGen = Lrm(augGr,1);
    std::cout << "*******************" << std::endl;
    std::cout << "Nonblock lrm" << std::endl;
    std::cout << "*******************" << std::endl;
    gotoGen.Write();
  }
  else{
    gotoGen = Lrm(augGr,1);
  }
  
  //construct a parser for the grammar
  Lr1Parser parser;
  if(print){
    parser = Lrp(gr, augGr, gotoGen, 1, t);
    std::cout << "*******************" << std::endl;
    std::cout << "Nonblock lrp" << std::endl;
    std::cout << "*******************" << std::endl;
    std::cout <<parser.Str() << std::endl;
  }
  else{
    parser = Lrp(gr, augGr, gotoGen, 1, t);
  }
  
  //convert parser to pushdown generator
  if(print){
    rPd = Gp2Pp(gotoGen, parser);
    std::cout << "*******************" << std::endl;
    std::cout << "Nonblock gp2pp" << std::endl;
    std::cout << "*******************" << std::endl;
    rPd.Write();
  }
  else{
    rPd = Gp2Pp(gotoGen, parser);
  }
  
  //remove augmentation
  if(print){
    rPd = Dim(rPd, t);
    std::cout << "*******************" << std::endl;
    std::cout << "Nonblock dim" << std::endl;
    std::cout << "*******************" << std::endl;
    rPd.Write();
  }
  else{
    rPd = Dim(rPd, t);
  }
  
  //make the pushdown generator accessible
  if(print){
    rPd.Accessible();
    std::cout << "*******************" << std::endl;
    std::cout << "Nonblock accessible" << std::endl;
    std::cout << "*******************" << std::endl;
    rPd.Write();
  }
  else{
    rPd.Accessible();
  }
  
  //split transitions popping more than one stack symbol into more transitions
  if(print){
    rPd = Rep2(rPd);
    std::cout << "*******************" << std::endl;
    std::cout << "Nonblock rep2" << std::endl;
    std::cout << "*******************" << std::endl;
    rPd.Write();
  }
  else{
    rPd = Rep2(rPd);
  }
  
  //remove transitions popping lambda
  if(print){
    rPd = Rep0(rPd);
    std::cout << "*******************" << std::endl;
    std::cout << "Nonblock rep0" << std::endl;
    std::cout << "*******************" << std::endl;
    rPd.Write();
  }
  else{
    rPd = Rep0(rPd);
  }
  
  
  //remove useless transitions created by rep0

  if(print){
    rPd = RemoveUselessTransitions(rPd);
    std::cout << "*******************" << std::endl;
    std::cout << "Nonblock removeUselessTransitions" << std::endl;
    std::cout << "*******************" << std::endl;
    rPd.Write();
  }
  else{
    rPd = RemoveUselessTransitions(rPd);
  }
  
  
  //set correct flags for controllability and observability
  CorrectEvents(pd, rPd);
  
  res = rPd;
}

/* *************************
 * CorrectEvents
 * *************************/
void CorrectEvents(const PushdownGenerator& correctPd, PushdownGenerator& pd){
  
  EventSet::Iterator correctevit, evit;
  for(evit = pd.AlphabetBegin(); evit != pd.AlphabetEnd(); evit++){
    for(correctevit = correctPd.AlphabetBegin(); correctevit != correctPd.AlphabetEnd(); correctevit++){
      if(*evit == *correctevit){
        //controllable 
        if(correctPd.Controllable(*correctevit)){
          pd.SetControllable(*evit);
        }
        //uncontrollable 
        else{
          pd.ClrControllable(*evit);
        }
        
        //observable 
        if(correctPd.Observable(*correctevit)){
          pd.SetObservable(*evit);
        }
        //uncontrollable 
        else{
          pd.ClrObservable(*evit);
        }
      }
    }
  }
}

/* *************************
 * Ts
 * *************************/
bool Ts(const PushdownGenerator& pd, Idx state){
  
  //bool to return
  bool reachable = false;
  
  //take copy of the generator
  PushdownGenerator tempPd = pd;
  
  //delete all final states and set state as new final state
  tempPd.ClearMarkedStates();
  tempPd.InsMarkedState(state);
  
  //make DFA-accessible to speed up nonblock
  tempPd.Accessible();
  tempPd = TrimStackSymbols(tempPd);

  //make the generator nonblocking
  
  /*uint stateCount = tempPd.Size();
  uint transCount = tempPd.TransRelSize();
  std::cout << "states: " << stateCount << ", transitions: " << transCount << std::endl;
  timeval start, finish, result;
  gettimeofday(&start,0);*/
  
  PushdownGenerator temp;
  PushdownNonblock(tempPd, temp);
  tempPd = temp;
  
  /*gettimeofday(&finish,0);
  timeval_subtract(&result, &finish, &start);
  float secstate = (float)result.tv_sec/(float)stateCount;
  float sectrans = (float)result.tv_sec/(float)transCount;
  std::cout << "time taken: "  << result.tv_sec << "." << result.tv_usec << " sec, " <<  secstate << " sec/state, " << sectrans << " sec/transition" <<  std::endl;*/
  
  //if the generator still has final states, the original state was reachable
  if(tempPd.MarkedStatesSize() > 0){
    reachable = true;
  }
  return reachable;
}

/* *************************
 * Te
 * *************************/
bool Te(const PushdownGenerator& pd, const Transition& trans, const std::vector<Idx>& pop, const std::vector<Idx>& push){
  
  //bool to return
  bool reachable = false;
  
  //take copy of the generator
  PushdownGenerator tempPd = pd;
  
  //remove the transition (q,x,w,w',q') from the generator
  tempPd.ClrTransition(trans,pop,push);
  
  //delete all final states and set state as new final state
  tempPd.ClearMarkedStates();
  Idx newState = tempPd.InsMarkedState();
  
  //insert transition (q,lambda,w,w,q'')
  tempPd.SetTransition(trans.X1,tempPd.EventIndex(FAUDES_PD_LAMBDA),newState,pop,pop);

  //make DFA-accessible to speed up nonblock
  tempPd.Accessible();
  tempPd = TrimStackSymbols(tempPd);

  
  
  //make the generator nonblocking
  
  /*uint stateCount = tempPd.Size();
  uint transCount = tempPd.TransRelSize();
  std::cout << "states: " << stateCount << ", transitions: " << transCount << std::endl;
  timeval start, finish, result;
  gettimeofday(&start,0);*/
  
  PushdownGenerator temp;
  PushdownNonblock(tempPd, temp);
  tempPd = temp;
  
  /*gettimeofday(&finish,0);
  timeval_subtract(&result, &finish, &start);
  float secstate = (float)result.tv_sec/(float)stateCount;
  float sectrans = (float)result.tv_sec/(float)transCount;
  std::cout << "time taken: "  << result.tv_sec << "." << result.tv_usec << " sec, " <<  secstate << " sec/state, " << sectrans << " sec/transition" <<  std::endl;*/
  
  //if the generator still has final states, the original state was reachable
  if(tempPd.MarkedStatesSize() > 0){
    reachable = true;
  }
  return reachable;
  
}

// /* *************************
//  * Ruls
//  * *************************/
// PushdownGenerator Ruls(const PushdownGenerator& pd){
//   
//   //generator to return
//   PushdownGenerator rPd = pd;
//   
//   //make DFA-Coaccessible to remove (hopefully a lot of) useless ears
//   rPd.Coaccessible();
//   
//   PushdownGenerator tempPd = rPd;
//   
//   StateSet earsToDelete;
//   StateSet::Iterator stateit;
//   //iterate over all states of the generator and delete certain ears
//   for(stateit = tempPd.StatesBegin(); stateit != tempPd.StatesEnd(); stateit++){
//     
//     //try to convert merge attribute of the state to MergeStateSplit
//     const MergeStateSplit* mss = dynamic_cast<const MergeStateSplit*>(tempPd.Merge(*stateit));
//     if(mss == NULL){
//       std::stringstream errstr;
//       errstr << "State in Ruls was neither head nor ear, check correctness of Merge attribute" << std::endl;
//       throw Exception("Ruls", errstr.str(), 1001);
//     }
//     
//     //only look at ears
//     if(mss->IsHead()){
//       continue;
//     }
//     
//     //do not delete ears that are at the start of a transition
//     if(tempPd.ExistsTransition(*stateit)){
//       continue;
//     }
//     
//     //do not delete final ears that are reachable
//     if(tempPd.ExistsMarkedState(*stateit)){
//       std::cout << "*** testing state " << *stateit << std::endl;
//       if(Ts(tempPd, *stateit)){
//         std::cout << "*** ... reachable" << std::endl;
//         continue;
//       }
//       else{
//           std::cout << "*** ... not reachable" << std::endl;
//       }
//     }
//     
//     //schedule all other ears for deletion
//     earsToDelete.Insert(*stateit);
//   }
//   
//   //delete ears
//   rPd.DelStates(earsToDelete);
//   
//   return rPd;
// }

/* *************************
 * Ruls1
 * *************************/
bool Ruls1(Idx state, const PushdownGenerator& pd){
  
  //try to convert merge attribute of the state to MergeStateSplit
    const MergeStateSplit* mss = dynamic_cast<const MergeStateSplit*>(pd.Merge(state));
    if(mss == NULL){
      std::stringstream errstr;
      errstr << "State in Ruls was neither head nor ear, check correctness of Merge attribute" << std::endl;
      throw Exception("Ruls", errstr.str(), 1001);
    }
    
    //only look at ears
    if(mss->IsHead()){
      return false;
    }
    
    //do not delete ears that are at the start of a transition
    if(pd.ExistsTransition(state)){
      return false;
    }
    
    //do not delete final ears that are reachable
    if(pd.ExistsMarkedState(state)){

      if(Ts(pd, state)){
        return false;
      }
      else{
        return true;
      }
    }
    //should never reach this
    return false;
}

/* *************************
 * Ruls
 * *************************/
PushdownGenerator Ruls(const PushdownGenerator& pd){
  
  //generator to return
  PushdownGenerator rPd = pd;
  
  //make DFA-Coaccessible to remove (hopefully a lot of) useless ears
  rPd.Coaccessible();
    
  StateSet statesToKeep;
  StateSet::Iterator stateit=rPd.StatesBegin();
  
  //keep checking states until all the generator's states are checked
  while(stateit!=rPd.StatesEnd()){
    //test for usefulness
    if (Ruls1(*stateit,rPd)){
      //delete state if not useful
      rPd.DelState(*(stateit++)); // fixed: 2013-12-17
    }else
    stateit++;
  }
  
  return rPd;
}

/* *************************
 * PushdownAccessible
 * *************************/
void PushdownAccessible(const PushdownGenerator& pd, PushdownGenerator& res){
 
  PushdownGenerator rPd = pd;
  
  //make DFA-accessible
  rPd.Accessible();
  //std::cout << "********* AC, after Accessible" << std::endl;
  //remove now-useless stack symbols
  rPd = TrimStackSymbols(rPd);
  //std::cout << "********* AC, after TrimStackSymbols" << std::endl;
  //rPd.Write();
  
  //rPd.Write();
  
  //iterate over all states to test DPA-accessibility
  StateSet states = rPd.States();
  StateSet::Iterator stateit=states.Begin();
  while(stateit != states.End()){    
    //disregard the start state
    if(rPd.ExistsInitState(*stateit)){
      stateit++;
      continue;
    }
    //test state for reachability
    if(!Ts(rPd, *stateit)){
      rPd.DelState(*(stateit++));
    }else (stateit++);
  }
  
  //iterate over the remaining transitions to test DPA-accessibility
  PushdownGenerator tempPd = rPd;
  TransSet::Iterator transit;
  PopPushSet::const_iterator ppit;
  for(transit = tempPd.TransRelBegin(); transit != tempPd.TransRelEnd(); transit++){
    for(ppit = tempPd.PopPushBegin(*transit); ppit != tempPd.PopPushEnd(*transit); ppit++){

      //test transition for reachability
      if(!Te(rPd, *transit, ppit->first, ppit->second)){
        rPd.ClrTransition(*transit, ppit->first, ppit->second);
      }
    }
  }
  
  res = rPd;
}

/* *************************
 * Ccs
 * *************************/
PushdownGenerator Ccs(const PushdownGenerator& contr, const System& plant){
  
  //generator to return
  PushdownGenerator rPd;

  //create product generator
  
  rPd = Times(plant,contr);

  //trim the generator
  PushdownGenerator temp;
  PushdownAccessible(rPd,temp);
  rPd = temp;
  
  //remove useless stack symbols
  rPd = TrimStackSymbols(rPd);
  
  //split states into heads and ears
  rPd = Split(rPd);
  
  //remove useless states
  rPd = Ruls(rPd);
  //save number of states
  uint numberOfStates = rPd.Size();

  //remove noncontrollable ears
  rPd = Rnce(rPd, plant);
  
  //if states were removed
  if(numberOfStates != rPd.Size()){
    PushdownGenerator res;
    PushdownNonblock(rPd, res, true);
    return res;
  }
  
  //else  return the input because no modification was required in the first place
  return contr;
}

/* *************************
 * Ccl
 * *************************/
PushdownGenerator Ccl(const PushdownGenerator& contr, const System& plant, Idx iteration){
  
  //generator to return
  PushdownGenerator rPd;
  
  //create controller candidate
  //create controller candidate
  rPd = Ccs(contr, plant);
  
  //termination condition: no states were removed
  if(rPd.Size() == contr.Size()){
    return contr;
  }
  //else reiterate
  rPd = Ccl(rPd, plant, iteration + 1);
  
  return rPd;
}

/* *************************
 * PushdownConstructController
 * *************************/
void PushdownConstructController(const PushdownGenerator& spec, const System& plant, PushdownGenerator& res){
  
  //product of specification and plant
  PushdownGenerator rPd = Times(plant, spec);
  
  //make nonblocking
  PushdownGenerator temp;
  PushdownNonblock(rPd,temp);
  rPd = temp;
  
  //construct controller
  rPd = Ccl(rPd, plant, 1);
  
  res = rPd;
}


/* *************************
 * timeval_subtract
 * *************************/
int timeval_subtract (timeval* result, timeval* x, timeval* y){
  /* Perform the carry for the later subtraction by updating y. */
  if (x->tv_usec < y->tv_usec) {
    int nsec = (y->tv_usec - x->tv_usec) / 1000000 + 1;
    y->tv_usec -= 1000000 * nsec;
    y->tv_sec += nsec;
  }
  if (x->tv_usec - y->tv_usec > 1000000) {
    int nsec = (x->tv_usec - y->tv_usec) / 1000000;
    y->tv_usec += 1000000 * nsec;
    y->tv_sec -= nsec;
  }

  /* Compute the time remaining to wait.
    tv_usec is certainly positive. */
  result->tv_sec = x->tv_sec - y->tv_sec;
  result->tv_usec = x->tv_usec - y->tv_usec;

  /* Return 1 if result is negative. */
  return x->tv_sec < y->tv_sec;
}
  
} // namespace faudes

