/** @file pd_alg_cc.h  Top-Level functions*/


/* Pushdown plugin for FAU Discrete Event Systems Library (libfaudes)

   Copyright (C) 2013  Stefan Jacobi, Sven Schneider, Anne-Kathrin Hess

*/


#ifndef FAUDES_PD_ALG_CC_H
#define FAUDES_PD_ALG_CC_H

#include "corefaudes.h"
#include "pd_pdgenerator.h"
#include "pd_alg_lrm.h"
#include "pd_alg_lrp.h"
#include "pd_alg_nb_sub_a.h"
#include "pd_alg_nb_sub_b.h"
#include "pd_lang_k.h"
#include <sys/time.h>

namespace faudes {
  
  /**
   * Get all states that are the starting state of a lambda reading transition
   * 
   * @param pd
   *    the generator
   * @return
   *    the states
   */
  StateSet Transient(const PushdownGenerator& pd);
  
  /**
   * Remove non-controllable ears from a generator. The initial state must not be an ear.
   * Times and Split must have been executed on the generator.
   * 
   * @param pd
   *    the pushdown generator with ears
   * @param s
   *    the system that was used in the intersection operation to generate the 
   * pushdown generator
   * @return
   *    pushdowngenerator without noncontrollable ears
   */
  PushdownGenerator Rnce(const PushdownGenerator& pd, const System& s);
  
  /**
   * synchronous product generator of a pushdown generator and a regular generator
   * 
   * @param reg
   *    the regualar generator
   * @param pd
   *    the pushdown generator
   * @return
   *    the synchronous product
   */
  PushdownGenerator Times(const System& reg, const PushdownGenerator& pd);
  
  /**
   * Adds the intersection of events of the first two generators to the 
   * result generator. The function considers observability and controllability,
   * so observability and controllability attributes will be kept.
   * 
   * @param s
   *    first generator
   * @param pd
   *    second generator
   * @param rPd
   *    result generator
   */
  void IntersectEvents(const System& s, const PushdownGenerator& pd, PushdownGenerator& rPd);
  
  /**
   * Splits the states of a pushdown generator into heads and ears. Each ear is
   * associated with a stack symbol.
   * 
   * Transitions will always be either from ear to head or from head to ear.
   * Transitions from ear to head will always pop the stack symbol associated
   * with the ear. Transitions from head to ear will always read lambda and 
   * pop and push the stack symbol associated with the ear.
   * 
   * @param pd
   *    pushdown generator to be split
   * @return
   *    the split pushdown generator
   */
  PushdownGenerator Split(const PushdownGenerator& pd);
  
  /**
   * Make a pushdown generator nonblocking. This will remodel the pushdown generator
   * and replace stack symbols with new ones.
   * 
   * @param pd
   *    pushdown generator to make nonblocking
   * @param res
   *    nonblocking pushdown generator
   * @param print
   *    temporary debug parameter
   */
  void PushdownNonblock(const PushdownGenerator& pd, PushdownGenerator& res, bool print = false);
  
  /**
   * Sets controllability and observability flags for a pushdown generator's events to 
   * the event flags of another pushdown generator
   * 
   * @param correctPd
   *    pushdown generator with correct flags
   * @param pd
   *    this generator's event flags will be set to the correctPd's event flags
   */
  void CorrectEvents(const PushdownGenerator& correctPd, PushdownGenerator& pd);
  
  /**
   * Test a state for reachability.
   * 
   * @param pd
   *    the pushdown generator
   * @param state
   *    the state to test for reachability
   * @return
   *    true if the state is reachable, else false
   */
  bool Ts(const PushdownGenerator& pd, Idx state);
  
  /**
   * Test a transition for reachability.
   * 
   * @param pd
   *    the pushdown generator
   * @param trans
   *    the transition to test for reachability
   * @param pop
   *    the pop of the transitions to test for reachability
   * @param push
   *    the push of the transitions to test for reachability
   * @return
   *    true if the transition is reachable, else false
   */
  bool Te(const PushdownGenerator& pd, const Transition& trans, const std::vector<Idx>& pop, const std::vector<Idx>& push);
  
  
  bool Ruls1(Idx state, const PushdownGenerator& pd);
  
  /**
   * Remove useless states of a pushdown generator.
   * 
   * @param pd
   *    the generator, which is required to have passed Split() before
   * @return
   *    generator that only has useable states.
   */
  PushdownGenerator Ruls(const PushdownGenerator& pd);
  

  
  /**
   * Construct the accessible part of a pushdown generator
   * 
   * @param pd
   *    pushdown generator to make accessible
   * @param res
   *    accessible pushdown generator
   */
  void PushdownAccessible(const PushdownGenerator& pd, PushdownGenerator& res);
  
  /**
   * Single execution of the loop to construct a minimal restrictive sound controller.
   * Alphabets of specification and plant are expected to be equal.
   * 
   * @param contr
   *    the controller candidate
   * @param plant
   *    the plant
   * @return
   *    product of controller candidate and plant
   */
  PushdownGenerator Ccs(const PushdownGenerator& contr, const System& plant);
  
  /**
   * Looped exectuion of to construct a minimal restrictive sound controller. Alphabets
   * of specification and plant are expected to be equal.
   * 
   * @param contr
   *    the controller candidate
   * @param plant
   *    the plant
   * @param iteration
   *    number of iterations of ccl
   * @return
   *    minimal restrictive controller
   */
  PushdownGenerator Ccl(const PushdownGenerator& contr, const System& plant, Idx iteration = 0);
  
  /**
   * Construct a minimal restrictive sound controller. Alphabets of specification and
   * plant are expected to be equal.
   * 
   * @param spec
   *    the specification
   * @param plant
   *    the plant
   * @param res
   *    the result
   * @return
   */
  void PushdownConstructController(const PushdownGenerator& spec, const System& plant, PushdownGenerator& res);
  
  /**
   * from http://www.gnu.org/software/libc/manual/html_node/Elapsed-Time.html
   * Calculates time difference x - y. For debugging and time measurement.
   * 
   * @param result
   *    the time difference is stored here
   * @param x
   *    the time x
   * @param y
   *    the time y
   * @return
   *    1 if x - y is negative, else 0
   */
  int timeval_subtract (timeval* result, timeval* x, timeval* y);




} // namespace faudes

#endif
