/** @file pd_alg_opt.cpp  functions for optimizations*/


/* Pushdown plugin for FAU Discrete Event Systems Library (libfaudes)

   Copyright (C) 2013  Stefan Jacobi, Sven Schneider, Anne-Kathrin Hess

*/

#include "pd_alg_opt.h"

namespace faudes {

/* *************************
 * TrimStackSymbols
 * *************************/
PushdownGenerator TrimStackSymbols(const PushdownGenerator& pd){
  
  //copy the generator
  PushdownGenerator rPd = pd;
  
  StackSymbolSet usedSymbols;
  TransSet::Iterator transit;
  PopPushSet::const_iterator ppit;
  std::vector<Idx>::const_iterator ssit;
  std::vector<Idx> pop, push;
  //iterate over all transitions to get all used stack symbols
  for(transit = pd.TransRelBegin(); transit != pd.TransRelEnd(); transit++){
    for(ppit = pd.PopPushBegin(*transit); ppit != pd.PopPushEnd(*transit); ppit++){
      
      //extract pop and push
      pop = ppit->first;
      push = ppit->second;
      
      //extract stack symbols
      for(ssit = pop.begin(); ssit != pop.end(); ssit++){
        usedSymbols.Insert(*ssit);
      }
      //extract stack symbols
      for(ssit = push.begin(); ssit != push.end(); ssit++){
        usedSymbols.Insert(*ssit);
      }
    }
  }
  
  //delete all stack symbols except for the used ones
  StackSymbolSet::Iterator currentssit;
  for(currentssit = pd.StackSymbolsBegin(); currentssit != pd.StackSymbolsEnd(); currentssit++){
    
    //stack symbol is not used and is not lambda
    if(!usedSymbols.Exists(*currentssit) && !pd.IsStackSymbolLambda(*currentssit)){
      rPd.DelStackSymbol(*currentssit);
    }
  }
  
  return rPd;
}

/* *************************
 * RemoveUselessTransitions
 * *************************/
PushdownGenerator RemoveUselessTransitions(const PushdownGenerator& pd){
  
  //generator to return
  PushdownGenerator rPd = pd;
  
  TransSet::Iterator transOutit, transInit;
  PopPushSet::const_iterator ppOutit, ppInit;
  //iterate over all transitions (q, a, uv, w, q')
  for(transOutit = pd.TransRelBegin(); transOutit != pd.TransRelEnd(); transOutit++){
    
    //get all possible stack tops for state q
    Idx q = transOutit->X1;
    StackSymbolSet possibleStackTops = GetPossibleStackTops(pd, StateSet(), q);
    
    //iterate over pop-push-sets
    for(ppOutit = pd.PopPushBegin(*transOutit); ppOutit != pd.PopPushEnd(*transOutit); ppOutit++){
      
      //get the symbol v that gets popped from the stack first
      Idx popSymbol = ppOutit->first.back();
      
      //if the current pop symbol is not a possible stack top, delete the current 
      //transition
      if(!possibleStackTops.Exists(popSymbol) && !pd.IsStackSymbolLambda(popSymbol)){
        rPd.ClrTransition(*transOutit, ppOutit->first, ppOutit->second);
      }
    }
  }
  
  //remove transition (p, lambda, u, u, q) that contribute nothing ("lambda read");
  
  //save generator for iterating
  PushdownGenerator tempPd = rPd;
  //save X2-sorted transition set for iterating
  TTransSet<TransSort::X2EvX1> transitionsByX2;
  rPd.TransRel().ReSort(transitionsByX2);
  //iterate over all transitions (p, lambda, u, u, q)
  for(transInit = tempPd.TransRelBegin(); transInit != tempPd.TransRelEnd(); transInit++){
    
    //the event must be lambda
    if(!rPd.IsEventLambda(transInit->Ev)){
      continue;
    }
    
    
    //iterate over pop push pairs
    for(ppInit = tempPd.PopPushBegin(*transInit); ppInit != tempPd.PopPushEnd(*transInit); ppInit++){
      
      //pop must equal push
      if(ppInit->first.size() != 1 || ppInit->first != ppInit->second){
        continue;
      }
      
      //get the target state q and stack symbol u
      Idx q = transInit->X2;
      Idx u = ppInit->first.front();
      
      //find all transitions (q, a, u, v, r) originating at q with pop u
      for(transOutit = tempPd.TransRelBegin(q); transOutit != tempPd.TransRelEnd(q); transOutit++){
        
        //the event must be lambda
//         if(!rPd.IsEventLambda(transOutit->Ev)){
//           continue;
//         }
        
        //iterate over pop push pairs
        for(ppOutit = tempPd.PopPushBegin(*transOutit); ppOutit != tempPd.PopPushEnd(*transOutit); ppOutit++){
          
          //test for u pop 
          if(ppOutit->first.size() == 1 && ppOutit->first.front() == u){
            
            //delete transition (p, lambda, u, u, q)
            rPd.ClrTransition(*transInit, ppInit->first, ppInit->second);
            
            //insert transition (p, a, u, v, r)
            rPd.SetTransition(transInit->X1, transOutit->Ev, transOutit->X2, ppOutit->first, ppOutit->second);
          }
        }
      }
    }
  }
  
  //there might be unreachable states and transitions left, so make the generator //accessible
  rPd.Accessible();
  
  return rPd;
}

/* *************************
 * GetPossibleStackTops
 * *************************/
StackSymbolSet GetPossibleStackTops(const PushdownGenerator& pd, const StateSet& examinedStates, Idx q){
  
  StackSymbolSet possibleStackTops;
  
  //if the current state has already been examined, it will contribute nothing new
  if(examinedStates.Exists(q)){
    return possibleStackTops;
  }
  
  //mark the current state as examined
  StateSet currentExaminedStates = examinedStates;
  currentExaminedStates.Insert(q);
  
  //if the current state is an init state, it will contribute at least the stack bottom
  if(pd.ExistsInitState(q)){
    possibleStackTops.Insert(pd.StackBottom());
  }
  
  //sort transitions by end state
  TTransSet<TransSort::X2EvX1> transByX2;
  pd.TransRel().ReSort(transByX2);
 
  TTransSet<TransSort::X2EvX1>::Iterator transit;
  PopPushSet::const_iterator ppit;
  //iterate over all transitions (p, b, u, w'v', q) going into q and collect all
  //stack symbols w'
  for(transit = transByX2.BeginByX2(q); transit != transByX2.EndByX2(q); transit++){
    for(ppit = pd.PopPushBegin(*transit); ppit != pd.PopPushEnd(*transit); ppit++){
      
      //get the symbol w' that gets pushed onto the stack last
      Idx pushSymbol = ppit->second.front();
      
      //if w' and u are lambda, get the transition's predecessor state stack tops
      if(pd.IsStackSymbolLambda(pushSymbol) && pd.IsStackSymbolLambda(ppit->first.front())){
        StackSymbolSet predStackTops = GetPossibleStackTops(pd, currentExaminedStates, transit->X1);
        possibleStackTops.SetUnion(predStackTops);
      }
      //if only w' is lambda, every stack symbol can be a predecessor, because it might
      //be under the stack top, about which nothing is known
      else if(pd.IsStackSymbolLambda(pushSymbol)){
        possibleStackTops.SetUnion(pd.StackSymbols());
        possibleStackTops.Erase(pd.StackSymbolIndex(FAUDES_PD_LAMBDA));
        break;
      }
      //if w' is not lambda, add w' to the possible stack tops
      else{
        possibleStackTops.Insert(pushSymbol);
      }
    }
  }
  
  return possibleStackTops;  
}
  
} // namespace faudes

