-- ftc_tutorial.lua
--
-- This Lua-script provides an application example for fault-hiding control 
-- via control reconfiguration.


-- ---------------- -- 
-- import test data --
-- ---------------- --
print()
print("######## Application Example for Fault-Hiding Control Reconfiguration")
print()
print("######## Import input data")
print()
local gLn = faudes.Generator("data/reconf_ln.gen")	-- nominal plant
local gEn = faudes.Generator("data/reconf_en.gen")	-- nominal specification
local gLf = faudes.Generator("data/reconf_lf.gen")    -- fault-accommodating plant
local gEf = faudes.Generator("data/reconf_ef.gen")    -- fault-accommodating spec
local gEr = faudes.Generator()                        -- reconfiguration spec
local gK = faudes.Generator()

-- elementary alphabets
local aUCON = faudes.Alphabet()	-- sensor events
local aCON = faudes.Alphabet()	-- actuator events
local aVUCON = faudes.Alphabet()	-- sensor events
local aVCON = faudes.Alphabet()	-- actuator events
local aHI = faudes.Alphabet()		-- high-level events
local aLO = faudes.Alphabet()		-- low-level events
local aF = faudes.Alphabet()		-- fault
local aC = faudes.Alphabet()
local aVC = faudes.Alphabet() 

print("######## Set up alphabets")
aUCON:FromString([[  <Alphabet> A1 A2 </Alphabet> ]])
aUCON:Name("Uncontrollable Events")
aCON:FromString([[  <Alphabet> a1 a2 </Alphabet> ]])
aCON:Name("Controllable Events")
aHI:FromString([[  <Alphabet> ah Ah </Alphabet> ]])
aHI:Name("High-Level Events")
aF:Insert("F")
aUCON:Write()
aCON:Write()
aHI:Write()
print()

-- ------------------------- --
-- nominal controller design -- 
-- ------------------------- --
print("######## Step 1: Nominal Controller Design")
gHn = faudes.Generator()	-- nominal Controller
faudes.FtcNomController(gLn,gEn,aCON,aUCON,aHI,gHn)
-- test dumps
FAUDES_TEST_DUMP("Nominal plant",gLn)
FAUDES_TEST_DUMP("Nominal controller",gHn)
gHn:SWrite()
print()

-- ------------------------------------- --
-- fault-accommodating controller design -- 
-- ------------------------------------- --
print("######## Step 1b: Fault-Accommodating Controller Design")
gHf = faudes.Generator()	-- fault-accommodating Controller
faudes.FtcFaController(gLf,gEf,aCON,aUCON,aHI,gHf)
-- test dumps
FAUDES_TEST_DUMP("Fault-Accommodating plant",gLf)
FAUDES_TEST_DUMP("Fault-Accommodating controller",gHf)
gHf:SWrite()
print()


-- ----------------------------- --
-- Compute Reconfiguration Block -- 
-- ----------------------------- --
print("######## Step 2: Reconfigurator Design")
gR = faudes.Generator()
-- set up inactivity conditions
faudes.FtcInactivityConditions(aCON,aUCON,gEr)
-- compute reconfigurator 
faudes.FtcReconfigurator(gLn,gLf,gEr,gEf,gHn,aCON,aUCON,aHI,aLO,gR)
gR:SWrite()
print()

-- check non-conflictingness
-- compute optimal self-reconfiguring closed-loop system
gHv = gHn:Copy()
faudes.FtcVirtualise(gHn,aHI,"_v",gHv)
faudes.Parallel(gHv,gR,gK)
faudes.Parallel(gK,gLf,gK)
print("######## Check non-conflictingness")
print(tostring(faudes.FtcIsNonconflicting(gK,aLO,aHI,aUCON,aVCON)))

-- test dumps
FAUDES_TEST_DUMP("Reconfiguration block",gR)

--
-- check self-reconfiguring closed-loop system for selected controllers
--
print("######## Check closed-loop system for selected controllers")
gHv1=faudes.Generator("data/reconf_hv1.gen")
gHv2=faudes.Generator("data/reconf_hv2.gen")
gK1 = faudes.Generator()
gK2 = faudes.Generator()
faudes.Parallel(gHv1,gR,gK1)
faudes.Parallel(gK1,gLf,gK1)
faudes.Parallel(gK,gHv2,gK2)
faudes.StateMin(gK1)
faudes.StateMin(gK2)
gK1:SetDefaultStateNames()
gK2:SetDefaultStateNames()


-- --------------- --
-- export          --
-- --------------- --
gLn:Write("tmp_ln.gen")
gEn:Write("tmp_en.gen")
gHn:Write("tmp_hn.gen")
gHf:Write("tmp_hf.gen")
gLf:Write("tmp_lfa.gen")
gEf:Write("tmp_efa.gen")
gR:Write("tmp_r.gen")
gEr:Write("tmp_er.gen")
gK1:Write("tmp_k1.gen")
gK2:Write("tmp_k2.gen")
gHv1:Write("tmp_hv1.gen")
gHv2:Write("tmp_hv2.gen")
gHv:Write("tmp_hv.gen")

-- 
-- check protocols
--
FAUDES_TEST_DIFF()
