/** @file diag_eventdiagnosis.h
Functions to check a system's diagnosability with respect to failure events (diagnosability and I-diagnosability) and computation of an event-diagnoser.
*/

#ifndef DIAG_EVENTDIAGNOSIS_H
#define DIAG_EVENTDIAGNOSIS_H

#include <vector>
#include "corefaudes.h"
#include "diag_debug.h"
#include "diag_generator.h"
#include "diag_attrdiagstate.h"
#include "diag_attrfailureevents.h"
#include "diag_attrfailuretypes.h"
#include "diag_attrlabelset.h"


namespace faudes {

/** @name Functions (diagnosability with respect to a failure partition) */
/** @{ doxygen group */

/**
Test a system's diagnosability with respect to a given failure partition.
Standard diagnosability of a plant is validated with respect to a given failure %partition. According to: "A Polynomial Algorithm for Testing Diagnosability of Discrete Event Systems" by Shengbing Jiang, Zhongdong Huang, Vigyan Chandra, and Ratnesh Kumar.
@param rGen
  Input generator, is a model of the original plant containing the relevant failures events.
@param rFailureTypeMap
  Failure %partition: maps failure type names to failure events.  
@param rReportString
  User-readable information of violating condition (in case of negative test result).
@ingroup DiagnosisPlugIn
*/
extern FAUDES_API bool IsEventDiagnosable(const System& rGen, const AttributeFailureTypeMap& rFailureTypeMap, std::string& rReportString);


/**
Test a system's I-diagnosability with respect to a given failure partition.
Validates I-Diagnosability with respect to a given failure and indicator %partition. The algorithm is a modification of IsDiagnosable().
@param rGen
  Input generator, is a model of the original plant containing the relevant failures events.
@param rFailureTypeMap
  Failure %partition: maps failure type names to failure events and indicator events.  
@param rReportString
  User-readable information of violating condition (in case of negative test result).
@ingroup DiagnosisPlugIn
*/
extern FAUDES_API bool IsIndicatorEventDiagnosable(const System& rGen, const AttributeFailureTypeMap& rFailureTypeMap, std::string& rReportString);

/**
Check if a generator meets the general assumptions of diagnosability as required by IsDiagnosable(const System&, const AttributeFailureTypeMap&, std::string&) and IsIndicatorDiagnosable(const System&, const AttributeFailureTypeMap&, std::string&).
It is examined if the following assumptions hold:
<ol>
  <li> The failure and indicator events are part of the generator's alphabet. </li>
  <li> The generator is live. </li>
  <li> There do not exist any cycles of unobservable events. </li> 
  <li> All failure events are unobservable. </li>
</ol>
@param rGen
  Input generator, is a model of the original plant containing the relevant failures events.
@param rFailureTypeMap
  Failure %partition: maps failure type names to failure events and indicator events.  
@param rReportString
  User-readable information of violationg condition (in case of negative result).
@return
  True if all assumptions are met. 
@exception Exception
  - A failure event does not form part of the alphabet of the generator (id 302).
  - A indicator event does not form part of the alphabet of the generator (id 303).  
@ingroup DiagnosisPlugIn
*/
extern FAUDES_API bool MeetsDiagnosabilityAssumptions(const System& rGen, const AttributeFailureTypeMap& rFailureTypeMap, std::string& rReportString);

/** @} doxygen group */


/**
Convert the reverse composition map of Parallel() by switching first and second entry.
@param rReverseCompositionMap
  Input map as generated by Parallel().
@param rCompositionMap
  Converted map.
*/
extern FAUDES_API void ConvertParallelCompositionMap(  const std::map<std::pair<Idx,Idx>,Idx>& rReverseCompositionMap, 
                        std::map<Idx,std::pair<Idx,Idx> >& rCompositionMap);
                        
/** 
Test if a generator is live.
@param rGen
  Input generator.
@param rReport
  User-readable information of violating condition (in case of negative result).
@return
  True if generator is live.
*/
extern FAUDES_API bool IsLive(const System &rGen, std::string& rReport);

/** 
Test if there exist any cycles of unobservable events in a generator.
@param rGen
  Input generator.
@param rReport
  User-readable information of violating condition (in case of positive test result).
@return
  True if there exists a cycle of unobservable events.
*/
extern FAUDES_API bool CycleOfUnobsEvents(const System& rGen, std::string& rReport);

/**
Check if all failure events are unobservable events in a generator's alphabet.
@param rGen
  Input generator, is a model of the original plant containing the relevant failures events.
@param rFailureTypeMap
  Failure %partition: maps failure type names to failure events and indicator events.  
@param rReport
  User-readable information of violating condition (in case of negative result).
@return
  True if all failures events are unobservable.
*/
extern FAUDES_API bool FailuresUnobservable(const System& rGen, const AttributeFailureTypeMap& rFailureTypeMap, std::string& rReport);

/** 
Test if there exist any cycles in a generator.
@param rGen
  Input generator.
@param rReport
  User-readable information of violating condition (in case of positive test result).
@return
  True if there exists a cycle.
*/
extern FAUDES_API bool ExistsCycle(const System& rGen, std::string& rReport);

/** 
Auxiliary function for ExistsCycle(const System&, std::string&). Starting from a state currState, this function makes a depth-first-search through the generator rGen.
@param rGen
  Input generator.
@param rTodo
  States to process.
@param currState
  Current state.
@param statesOnPath
  States that occurred on the way to current state.
@param rReport
  User-readable information of violating condition (in case of negative test result).
@return
  True if a cycle is found.
*/
extern FAUDES_API bool ExistsCycleSearch(const System& rGen, StateSet& rTodo, Idx currState, StateSet statesOnPath, std::string& rReport);

/** 
Find all start/end states of cycles of unobservable events in a generator. 
@param rGen
  Input generator.
@param rCycleOrigins
  Output variable for the states that have been found.
*/
extern FAUDES_API void CycleStartStates(const System& rGen, StateSet& rCycleOrigins);

/** 
Auxiliary function for CycleStartStates(). Parses through the active event set of the current state and checks whether any of the successor states already occurred on the path to current state (then a cycle is found).
@param rGen
  Input generator.
@param rTodo
  States that remain to be processed.
@param currState
  The current state.
@param statesOnPath
  States that occurred on the path.
@param rCycleOriginStates
  Output variable for the states that have been found.
*/
extern FAUDES_API void CycleStartStatesSearch(const System& rGen, StateSet& rTodo, Idx currState, StateSet statesOnPath, StateSet& rCycleOriginStates);

/**
Remove states with same failure labels from rGd and from rReverseCompositionMap and perform cycle detection.
@param rGd
  Input diagnoser generator.
@param rGobs
  Generator G_o to look up failure labels.  
@param rReverseCompositionMap
  Mapping of G_d states with G_o states.
@param rFailureType
  The considered failure type.
@param rReportString
  User-readable information of violating condition (in case of negative test result).
@return
  True if violating cycles exist.
*/
extern FAUDES_API bool ExistsViolatingCyclesInGd(System& rGd, const Diagnoser& rGobs, std::map<std::pair<Idx,Idx>,Idx>& rReverseCompositionMap, const std::string& rFailureType, std::string& rReportString);

/** 
Compute G_o for a single failure type of a generator.
@param rOrigGen
  Input generator, which is a model of the original plant containing the relevant failures events.
@param rFailureType
  Failure type name.
@param rFailureEvents
  Failure events belonging to the failure type.
@param rGobs
  Output variable for G_o.
*/
extern FAUDES_API void ComputeGobs(const System& rOrigGen, const std::string& rFailureType, const EventSet& rFailureEvents, Diagnoser& rGobs);

/**
Compute G_o for a given generator with a given failure %partition (according to Jiang).
@param rOrigGen
  Input generator, which is a model of the original plant containing the relevant failures events.
@param rAttrFTMap
  Failure %partition.
@param rGobs
  Output variable for G_o. 
@exception Exception
  - Input generator has no unique initial state (id 301).
*/
extern FAUDES_API void ComputeGobs(const System& rOrigGen, const AttributeFailureTypeMap& rAttrFTMap, Diagnoser& rGobs);

/**
Compute the diagnosability testing generator G_d as a parallel composition of G_o with itself (according to Jiang).
@param rGobs
  Input diagnoser G_o.
@param rReverseCompositionMap
  Output variable containing the mapping of G_o states and G_d states (generated by Parallel()).
@param rGd
  Output variable for G_d.
*/
extern FAUDES_API void ComputeGd(const Diagnoser& rGobs, std::map<std::pair<Idx,Idx>,Idx>& rReverseCompositionMap, System& rGd);

/**
Extract all traces of a generator G_d that start with an indicator event that follows a failure event of a certain failure type.
@param rGd
  Input generator G_d which will be prunded.
@param rGobs
  Generator G_o (containing failure label information).
@param rFailureType
  Failure type index.
@param rIndicatorEvents
  Set of indicator events.
@param rReverseCompositionMap
  Mapping of G_d states with G_o states.
*/
extern FAUDES_API void TrimNonIndicatorTracesOfGd(System& rGd, const Diagnoser& rGobs, const Idx rFailureType,
                const EventSet& rIndicatorEvents, const std::map<std::pair<Idx,Idx>,Idx>& rReverseCompositionMap);

/**
Auxiliary function for TrimNonIndicatorTracesOfGd(). Parse through active transition set of the current state and delete transitions if occurring event is not an indicator event or there did not occur a failure before the indicator.
@param rGd
  Input generator G_d which will be pruned.
@param rGobs
  Generator G_o (containing failure label information).
@param rFailureType
  Failure type index.
@param rIndicatorEvents
  Set of indicator events.
@param rCompositionMap
  Inverted mapping of G_d states with G_o states.
@param state
  Current state.
@param rStatesDone
  States that have been processed already.
*/                
extern FAUDES_API void TrimNonIndicatorTracesOfGdRecursive(System& rGd, const Diagnoser& rGobs, const Idx rFailureType, 
                            const EventSet& rIndicatorEvents, std::map<Idx,std::pair<Idx,Idx> >& rCompositionMap,
                            Idx state, StateSet& rStatesDone);

/**
Compute the reachability from a given generator state through a trace that consists of arbitrarily many unobservable events followed by one observable event. 
@param rGen
  Input generator.
@param rUnobsEvents
  Unobservable events in the generators alphabet.
@param rFailures
  Unobservable failure events in the generators alphabet.
@param State
  A state of the generators state set.
@param rAttrFTMap
  Failure %partition.
@param rReachabilityMap
  Output variable for the reachability. Maps occurring observable events to the reachable generator states and the corresponding failure types that occurred within the unobservable part of the trace.
*/
extern FAUDES_API void ComputeReachability(const System& rGen, const EventSet& rUnobsEvents, const EventSet& rFailures, Idx State, 
                const AttributeFailureTypeMap& rAttrFTMap, std::map<Idx,std::multimap<Idx,DiagLabelSet> >& rReachabilityMap);

/**
Auxiliary function for ComputeReachability(const System&, const EventSet&, const EventSet&, Idx, const AttributeFailureTypeMap&, std::map<Idx,std::multimap<Idx,DiagLabelSet>>&). Is recursively called for every occurring state on the trace (that consists of arbitrarily many unobservable events followed by one observable event).
@param rGen
  Input generator.
@param rUnobsEvents
  Unobservable events in the generators alphabet.
@param rFailures
  Unobservable failure events in the generators alphabet.
@param State
  The current state within the trace.
@param rAttrFTMap
  Failure %partition.
@param rReachabilityMap
  Output variable for the reachability. Maps occurring observable events to the reachable generator states and the coresponing failure types that occurred within the unobservable part of the trace.
@param FToccurred
  Collects occurring failure types.
*/                
extern FAUDES_API void ComputeReachabilityRecursive(const System& rGen, const EventSet& rUnobsEvents,
                      const EventSet& rFailures, Idx State, const AttributeFailureTypeMap& rAttrFTMap,
                      std::map<Idx,std::multimap<Idx,DiagLabelSet> >& rReachabilityMap, const DiagLabelSet FToccurred);

/**
Obtain all transitions from other states into a given state of a generator.
@param rGen
  A generator.
@param state
  A state from the generators state set.
@return
  A transition set.
*/                      
extern FAUDES_API TransSet ActiveBackwardTransSet(const System& rGen, Idx state);


/**
Compute a standard diagnoser from an input generator and a failure partition.
@param rOrigGen
  Input plant including failure events.
@param rFailureTypeMap
  Failure %partition: maps failure type names to failure events.
@param rDiagGen
  Diagnoser generator for output.
*/
extern FAUDES_API void EventDiagnoser(const System& rOrigGen, const std::map<std::string,EventSet>& rFailureTypeMap, Diagnoser& rDiagGen);

/** @name Functions (diagnoser computation) */
/** @{ doxygen group */

/**
Compute a standard diagnoser from an input generator and a failure partition.
@param rOrigGen
  Input plant including failure events.
@param rAttrFTMap
  Failure type map of plant.
@param rDiagGen
  Diagnoser generator for output. 
@exception Exception
  - Input generator has no unique initial state (id 301).
@ingroup DiagnosisPlugIn
*/
extern FAUDES_API void EventDiagnoser(const System& rOrigGen, const AttributeFailureTypeMap& rAttrFTMap, Diagnoser& rDiagGen);

/** @} doxygen group */

/**
Generate a new label. From the last label and the failure types that occurred on the way, the new label is generated.
@param lastLabel
  Diagnoser label of state estimate.
@param failureTypes
  Failure types that occurred.
@param newLabel
  Output variable for the new label.
*/
extern FAUDES_API void LabelPropagation(const DiagLabelSet& lastLabel, const DiagLabelSet& failureTypes, DiagLabelSet& newLabel);

/**
Perform label correction on propagated failure type labels.
@param mm
  Propagated absolute failure type labels for a diagnoser state.
@param attr
  Output variable for the diagnoser state attribute.
*/
extern FAUDES_API void LabelCorrection(const std::multimap<Idx,DiagLabelSet>& mm, AttributeDiagnoserState& attr);

/**
 * Function definition for run-time interface 
 */
extern FAUDES_API bool IsEventDiagnosable(const System& rGen, const AttributeFailureTypeMap& rFailureTypeMap);


/**
 * Function definition for run-time interface 
 */
extern FAUDES_API bool IsIndicatorEventDiagnosable(const System& rGen, const AttributeFailureTypeMap& rFailureTypeMap);


} // namespace faudes

#endif
